/**
 * Modern Scroll Utilities v2.0
 * Clean implementation without deprecated event warnings
 */

(function() {
    'use strict';
    
    // Silent initialization without console logs
    const ModernScrollManager = {
        init() {
            this.setupIntersectionObserver();
            this.setupSmoothScrolling();
            this.blockDeprecatedEvents();
        },
        
        setupIntersectionObserver() {
            if (!window.IntersectionObserver) return;
            
            const observer = new IntersectionObserver((entries) => {
                entries.forEach(entry => {
                    if (entry.isIntersecting) {
                        entry.target.classList.add('in-view');
                    } else {
                        entry.target.classList.remove('in-view');
                    }
                });
            }, {
                rootMargin: '50px',
                threshold: 0.1
            });
            
            // Observe animation elements
            document.querySelectorAll('[data-scroll], .fade-in, .slide-up, .card').forEach(el => {
                observer.observe(el);
            });
        },
        
        setupSmoothScrolling() {
            document.addEventListener('click', (e) => {
                const link = e.target.closest('a[href^="#"]');
                if (!link) return;
                
                const target = document.querySelector(link.getAttribute('href'));
                if (target) {
                    e.preventDefault();
                    target.scrollIntoView({ behavior: 'smooth' });
                }
            });
        },
        
        blockDeprecatedEvents() {
            const original = EventTarget.prototype.addEventListener;
            EventTarget.prototype.addEventListener = function(type, listener, options) {
                if (['DOMNodeInserted', 'DOMNodeRemoved', 'DOMSubtreeModified'].includes(type)) {
                    return; // Silently block
                }
                return original.call(this, type, listener, options);
            };
        }
    };
    
    // Add minimal CSS
    if (!document.getElementById('modern-scroll-css')) {
        const style = document.createElement('style');
        style.id = 'modern-scroll-css';
        style.textContent = `
            [data-scroll], .fade-in, .slide-up {
                opacity: 0;
                transform: translateY(20px);
                transition: opacity 0.6s ease, transform 0.6s ease;
            }
            [data-scroll].in-view, .fade-in.in-view, .slide-up.in-view {
                opacity: 1;
                transform: translateY(0);
            }
            @media (prefers-reduced-motion: reduce) {
                [data-scroll], .fade-in, .slide-up { transition: none; }
            }
        `;
        document.head.appendChild(style);
    }
    
    // Initialize
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', () => ModernScrollManager.init());
    } else {
        ModernScrollManager.init();
    }
    
    window.ModernScroll = ModernScrollManager;
    
})(); 