const CACHE_NAME = 'portal-desa-v1.0.2';
const STATIC_CACHE = 'portal-desa-static-v1.0.2';
const DYNAMIC_CACHE = 'portal-desa-dynamic-v1.0.2';

// Assets to cache immediately - only files that definitely exist
const STATIC_ASSETS = [
    '/',
    '/manifest.json',
    '/offline.html'
];

// Optional assets to try caching (won't fail if they don't exist)
const OPTIONAL_ASSETS = [
    '/css/app.css',
    '/js/app.js',
    '/build/assets/app-ewwVRAv0.css',
    '/build/assets/app-BjIuunQC.js'
];

// Runtime caching patterns
const CACHE_STRATEGIES = {
    pages: 'networkFirst',
    assets: 'cacheFirst',
    api: 'networkFirst'
};

// Install event - cache static assets with error handling
self.addEventListener('install', event => {
    console.log('SW: Installing...');
    
    event.waitUntil(
        Promise.all([
            // Cache essential assets (must succeed)
            caches.open(STATIC_CACHE)
                .then(cache => {
                    console.log('SW: Caching essential assets');
                    return cache.addAll(STATIC_ASSETS);
                }),
            
            // Cache optional assets (can fail individually)
            cacheOptionalAssets()
        ])
        .then(() => {
            console.log('SW: All assets cached successfully');
            return self.skipWaiting();
        })
        .catch(error => {
            console.error('SW: Cache failed', error);
            // Continue installation even if some assets fail
            return self.skipWaiting();
        })
    );
});

// Cache optional assets with individual error handling
async function cacheOptionalAssets() {
    const cache = await caches.open(STATIC_CACHE);
    
    // Try to cache each optional asset individually
    const promises = OPTIONAL_ASSETS.map(async (url) => {
        try {
            const response = await fetch(url);
            if (response.ok) {
                await cache.put(url, response);
                console.log('SW: Cached optional asset:', url);
            }
        } catch (error) {
            console.log('SW: Failed to cache optional asset:', url, error.message);
            // Don't throw - just log and continue
        }
    });
    
    await Promise.allSettled(promises);
}

// Activate event - cleanup old caches
self.addEventListener('activate', event => {
    console.log('SW: Activating...');
    
    event.waitUntil(
        caches.keys()
            .then(cacheNames => {
                return Promise.all(
                    cacheNames
                        .filter(cacheName => {
                            return cacheName.startsWith('portal-desa-') && 
                                   cacheName !== STATIC_CACHE && 
                                   cacheName !== DYNAMIC_CACHE;
                        })
                        .map(cacheName => {
                            console.log('SW: Deleting old cache', cacheName);
                            return caches.delete(cacheName);
                        })
                );
            })
            .then(() => {
                console.log('SW: Activated');
                return self.clients.claim();
            })
    );
});

// Fetch event - handle requests with caching strategies
self.addEventListener('fetch', event => {
    const url = new URL(event.request.url);
    
    // Skip non-GET requests
    if (event.request.method !== 'GET') {
        return;
    }
    
    // Skip external requests
    if (url.origin !== location.origin) {
        return;
    }
    
    event.respondWith(handleRequest(event.request));
});

async function handleRequest(request) {
    const url = new URL(request.url);
    
    try {
        // API requests - Network first with fallback
        if (url.pathname.startsWith('/api/')) {
            return await networkFirst(request, DYNAMIC_CACHE);
        }
        
        // Static assets - Cache first
        if (isStaticAsset(url.pathname)) {
            return await cacheFirst(request, STATIC_CACHE);
        }
        
        // Pages - Network first with offline fallback
        if (isPageRequest(request)) {
            return await networkFirstWithOffline(request, DYNAMIC_CACHE);
        }
        
        // Default to network first
        return await networkFirst(request, DYNAMIC_CACHE);
        
    } catch (error) {
        console.error('SW: Request failed', error);
        return getOfflinePage();
    }
}

// Cache first strategy
async function cacheFirst(request, cacheName) {
    const cache = await caches.open(cacheName);
    const cached = await cache.match(request);
    
    if (cached) {
        // Update cache in background
        updateCache(request, cacheName);
        return cached;
    }
    
    return await fetchAndCache(request, cacheName);
}

// Network first strategy
async function networkFirst(request, cacheName) {
    try {
        const response = await fetch(request);
        
        if (response.ok) {
            const cache = await caches.open(cacheName);
            cache.put(request, response.clone());
        }
        
        return response;
    } catch (error) {
        const cache = await caches.open(cacheName);
        const cached = await cache.match(request);
        
        if (cached) {
            return cached;
        }
        
        throw error;
    }
}

// Network first with offline page fallback
async function networkFirstWithOffline(request, cacheName) {
    try {
        return await networkFirst(request, cacheName);
    } catch (error) {
        return getOfflinePage();
    }
}

// Fetch and cache helper with better error handling
async function fetchAndCache(request, cacheName) {
    try {
        const response = await fetch(request);
        
        if (response.ok) {
            const cache = await caches.open(cacheName);
            await cache.put(request, response.clone());
        }
        
        return response;
    } catch (error) {
        console.error('SW: Failed to fetch and cache:', request.url, error);
        throw error;
    }
}

// Update cache in background
function updateCache(request, cacheName) {
    fetch(request)
        .then(response => {
            if (response.ok) {
                caches.open(cacheName)
                    .then(cache => cache.put(request, response));
            }
        })
        .catch(() => {
            // Silently fail background updates
        });
}

// Get offline page
async function getOfflinePage() {
    const cache = await caches.open(STATIC_CACHE);
    const offlinePage = await cache.match('/offline.html');
    
    return offlinePage || new Response('Offline - Please check your connection', {
        status: 503,
        statusText: 'Service Unavailable',
        headers: {
            'Content-Type': 'text/plain'
        }
    });
}

// Helper functions
function isStaticAsset(pathname) {
    return pathname.match(/\.(css|js|png|jpg|jpeg|gif|svg|ico|woff|woff2|ttf|eot)$/);
}

function isPageRequest(request) {
    return request.headers.get('accept')?.includes('text/html');
}

// Background sync for offline actions
self.addEventListener('sync', event => {
    console.log('SW: Background sync', event.tag);
    
    if (event.tag === 'offline-submissions') {
        event.waitUntil(syncOfflineSubmissions());
    }
});

async function syncOfflineSubmissions() {
    try {
        // Get offline submissions from IndexedDB
        const submissions = await getOfflineSubmissions();
        
        for (const submission of submissions) {
            try {
                const response = await fetch(submission.url, {
                    method: submission.method,
                    headers: submission.headers,
                    body: submission.body
                });
                
                if (response.ok) {
                    // Remove successful submission
                    await removeOfflineSubmission(submission.id);
                    
                    // Notify user of success
                    showNotification('Pengajuan berhasil dikirim!', {
                        body: 'Data yang tersimpan offline telah berhasil dikirim.',
                        icon: '/icon-192.png',
                        badge: '/icon-96.png'
                    });
                }
            } catch (error) {
                console.error('SW: Sync failed for submission', submission.id, error);
            }
        }
    } catch (error) {
        console.error('SW: Background sync failed', error);
    }
}

// Push notifications
self.addEventListener('push', event => {
    if (!event.data) return;
    
    try {
        const data = event.data.json();
        
        event.waitUntil(
            showNotification(data.title, {
                body: data.body,
                icon: data.icon || '/icon-192.png',
                badge: '/icon-96.png',
                tag: data.tag || 'portal-desa',
                data: data.data || {},
                actions: data.actions || []
            })
        );
    } catch (error) {
        console.error('SW: Push notification failed', error);
    }
});

// Notification click handler
self.addEventListener('notificationclick', event => {
    event.notification.close();
    
    const action = event.action;
    const data = event.notification.data;
    
    event.waitUntil(
        clients.matchAll({ includeUncontrolled: true })
            .then(clients => {
                // Focus existing tab or open new one
                if (clients.length > 0) {
                    const client = clients[0];
                    client.focus();
                    
                    if (data.url) {
                        client.navigate(data.url);
                    }
                } else {
                    self.clients.openWindow(data.url || '/');
                }
            })
    );
});

// Utility functions for notifications
async function showNotification(title, options) {
    const registration = await self.registration;
    return registration.showNotification(title, {
        vibrate: [200, 100, 200],
        requireInteraction: false,
        ...options
    });
}

// IndexedDB helpers for offline storage
async function getOfflineSubmissions() {
    // Implement IndexedDB operations for offline submissions
    return [];
}

async function removeOfflineSubmission(id) {
    // Implement IndexedDB remove operation
}

// Message handling from main thread
self.addEventListener('message', event => {
    if (event.data && event.data.type) {
        switch (event.data.type) {
            case 'SKIP_WAITING':
                self.skipWaiting();
                break;
            case 'GET_VERSION':
                event.ports[0].postMessage({ version: CACHE_NAME });
                break;
            case 'CACHE_URLS':
                cacheUrls(event.data.urls);
                break;
        }
    }
});

async function cacheUrls(urls) {
    const cache = await caches.open(DYNAMIC_CACHE);
    await cache.addAll(urls);
}

console.log('SW: Script loaded'); 