<?php
/**
 * Test Edit Form and Update Process
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use App\Models\User;
use App\Services\FileUploadService;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Test Edit Form & Update</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 1000px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .form-control { width: 100%; padding: 8px; border: 1px solid #ddd; border-radius: 4px; margin: 5px 0; }
        .btn { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; margin: 5px; }
        .btn-success { background: #28a745; }
        .btn-danger { background: #dc3545; }
        .test-form { background: #f8f9ff; border: 2px solid #007bff; padding: 20px; border-radius: 8px; margin: 20px 0; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🔧 Test Edit Form & Update Process</h1>
        
        <?php
        // Handle form submission
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            echo "<div class='section'>";
            echo "<h2>📤 Form Submission Test</h2>";
            
            try {
                // Get tourism object
                $tourismId = $_POST['tourism_id'] ?? 1;
                $tourism = ObjekWisata::find($tourismId);
                
                if (!$tourism) {
                    throw new Exception("Tourism with ID {$tourismId} not found");
                }
                
                echo "<div class='info'>🎯 Processing tourism: {$tourism->nama_wisata}</div>";
                
                // Validate required fields
                $required = ['nama_wisata', 'kategori_wisata', 'deskripsi', 'alamat_lengkap', 'harga_tiket_dewasa'];
                $errors = [];
                
                foreach ($required as $field) {
                    if (empty($_POST[$field])) {
                        $errors[] = "Field {$field} is required";
                    }
                }
                
                if (!empty($errors)) {
                    foreach ($errors as $error) {
                        echo "<div class='error'>❌ {$error}</div>";
                    }
                } else {
                    echo "<div class='success'>✅ All required fields present</div>";
                    
                    // Test file upload if present
                    $newPhotoFile = null;
                    if (isset($_FILES['foto_utama']) && $_FILES['foto_utama']['error'] === UPLOAD_ERR_OK) {
                        echo "<div class='info'>📁 Processing file upload...</div>";
                        
                        $uploadedFile = new \Illuminate\Http\UploadedFile(
                            $_FILES['foto_utama']['tmp_name'],
                            $_FILES['foto_utama']['name'],
                            $_FILES['foto_utama']['type'],
                            $_FILES['foto_utama']['error'],
                            true
                        );
                        
                        // Test file validation
                        $allowedTypes = ['image/jpeg', 'image/png', 'image/jpg', 'image/webp'];
                        if (!in_array($uploadedFile->getMimeType(), $allowedTypes)) {
                            echo "<div class='error'>❌ Invalid file type: {$uploadedFile->getMimeType()}</div>";
                        } elseif ($uploadedFile->getSize() > 2048 * 1024) {
                            echo "<div class='error'>❌ File too large: " . round($uploadedFile->getSize() / 1024) . "KB</div>";
                        } else {
                            echo "<div class='success'>✅ File validation passed</div>";
                            
                            // Upload file
                            $uploadService = new FileUploadService();
                            $uploadResult = $uploadService->uploadPhotoWithResize($uploadedFile, 'wisata');
                            
                            if ($uploadResult['success']) {
                                $newPhotoFile = $uploadResult['filename'];
                                echo "<div class='success'>✅ File uploaded: {$newPhotoFile}</div>";
                            } else {
                                echo "<div class='error'>❌ Upload failed: {$uploadResult['error']}</div>";
                            }
                        }
                    }
                    
                    // Update tourism object
                    $updateData = [
                        'nama_wisata' => $_POST['nama_wisata'],
                        'kategori_wisata' => $_POST['kategori_wisata'],
                        'deskripsi' => $_POST['deskripsi'],
                        'alamat' => $_POST['alamat_lengkap'],
                        'harga_tiket_dewasa' => floatval($_POST['harga_tiket_dewasa']),
                    ];
                    
                    if ($newPhotoFile) {
                        $updateData['foto_utama'] = $newPhotoFile;
                    }
                    
                    // Add optional fields
                    if (!empty($_POST['harga_tiket_anak'])) {
                        $updateData['harga_tiket_anak'] = floatval($_POST['harga_tiket_anak']);
                    }
                    if (!empty($_POST['latitude'])) {
                        $updateData['latitude'] = floatval($_POST['latitude']);
                    }
                    if (!empty($_POST['longitude'])) {
                        $updateData['longitude'] = floatval($_POST['longitude']);
                    }
                    
                    $result = $tourism->update($updateData);
                    
                    if ($result) {
                        echo "<div class='success'>✅ Tourism updated successfully!</div>";
                        
                        // Show updated data
                        $updatedTourism = $tourism->fresh();
                        echo "<div class='info'><strong>Updated data:</strong></div>";
                        echo "<div class='code'>";
                        echo "Name: {$updatedTourism->nama_wisata}\n";
                        echo "Category: {$updatedTourism->kategori_wisata}\n";
                        echo "Address: {$updatedTourism->alamat}\n";
                        echo "Price: Rp " . number_format($updatedTourism->harga_tiket_dewasa) . "\n";
                        echo "Photo: " . ($updatedTourism->foto_utama ?: 'NULL') . "\n";
                        echo "Updated: {$updatedTourism->updated_at}";
                        echo "</div>";
                        
                        // Test URL generation
                        if ($updatedTourism->foto_utama) {
                            $imageUrl = $updatedTourism->foto_utama_url;
                            echo "<div class='info'><strong>Image URL:</strong> {$imageUrl}</div>";
                            echo "<img src='{$imageUrl}?v=" . time() . "' style='max-width: 200px; border: 1px solid #ddd; margin: 10px;' alt='Updated Image'>";
                        }
                        
                    } else {
                        echo "<div class='error'>❌ Tourism update failed</div>";
                    }
                }
                
            } catch (Exception $e) {
                echo "<div class='error'>❌ Exception: " . $e->getMessage() . "</div>";
            }
            echo "</div>";
        }
        
        // Get tourism data for form
        $tourism = ObjekWisata::first();
        
        if (!$tourism) {
            echo "<div class='error'>❌ No tourism data found</div>";
            exit;
        }
        
        echo "<div class='section'>";
        echo "<h2>🎯 Current Tourism Data</h2>";
        echo "<div class='info'><strong>ID:</strong> {$tourism->id}</div>";
        echo "<div class='info'><strong>Name:</strong> {$tourism->nama_wisata}</div>";
        echo "<div class='info'><strong>Category:</strong> {$tourism->kategori_wisata}</div>";
        echo "<div class='info'><strong>Current Photo:</strong> " . ($tourism->foto_utama ?: 'NULL') . "</div>";
        
        if ($tourism->foto_utama) {
            $currentUrl = $tourism->foto_utama_url;
            echo "<div class='info'><strong>Current URL:</strong> {$currentUrl}</div>";
            echo "<img src='{$currentUrl}?v=" . time() . "' style='max-width: 200px; border: 1px solid #ddd; margin: 10px;' alt='Current Image'>";
        }
        echo "</div>";
        
        // Check admin users
        echo "<div class='section'>";
        echo "<h2>👤 Admin Users Check</h2>";
        
        $adminUsers = User::where('role', 'admin')->get();
        echo "<div class='info'>📊 Admin users found: " . $adminUsers->count() . "</div>";
        
        foreach ($adminUsers->take(3) as $user) {
            echo "<div class='info'>• {$user->name} ({$user->email})</div>";
        }
        
        if ($adminUsers->count() === 0) {
            echo "<div class='warning'>⚠️ No admin users found. You need to create admin account first.</div>";
        }
        echo "</div>";
        
        // Check directories
        echo "<div class='section'>";
        echo "<h2>📁 Directory Check</h2>";
        
        $directories = [
            'storage/app/public/wisata' => storage_path('app/public/wisata'),
            'public/storage/wisata' => public_path('storage/wisata'),
        ];
        
        foreach ($directories as $name => $path) {
            $exists = is_dir($path);
            $writable = $exists && is_writable($path);
            $fileCount = $exists ? count(glob($path . '/*')) : 0;
            
            echo "<div class='" . ($writable ? 'success' : 'error') . "'>";
            echo "<strong>{$name}:</strong> ";
            echo ($exists ? '📁 EXISTS' : '❌ NOT FOUND') . " | ";
            echo ($writable ? '✅ WRITABLE' : '❌ NOT WRITABLE') . " | ";
            echo "📄 {$fileCount} files";
            echo "</div>";
        }
        echo "</div>";
        ?>
        
        <!-- Test Form -->
        <div class="test-form">
            <h2>📝 Test Update Form</h2>
            <p><strong>This form simulates the admin edit process:</strong></p>
            
            <form method="POST" enctype="multipart/form-data">
                <input type="hidden" name="tourism_id" value="<?= $tourism->id ?>">
                
                <div>
                    <label><strong>Tourism Name:</strong></label>
                    <input type="text" name="nama_wisata" class="form-control" value="<?= htmlspecialchars($tourism->nama_wisata) ?>" required>
                </div>
                
                <div>
                    <label><strong>Category:</strong></label>
                    <select name="kategori_wisata" class="form-control" required>
                        <option value="wisata_alam" <?= $tourism->kategori_wisata === 'wisata_alam' ? 'selected' : '' ?>>Wisata Alam</option>
                        <option value="wisata_budaya" <?= $tourism->kategori_wisata === 'wisata_budaya' ? 'selected' : '' ?>>Wisata Budaya</option>
                        <option value="wisata_sejarah" <?= $tourism->kategori_wisata === 'wisata_sejarah' ? 'selected' : '' ?>>Wisata Sejarah</option>
                        <option value="lainnya" <?= $tourism->kategori_wisata === 'lainnya' ? 'selected' : '' ?>>Lainnya</option>
                    </select>
                </div>
                
                <div>
                    <label><strong>Description:</strong></label>
                    <textarea name="deskripsi" class="form-control" required><?= htmlspecialchars($tourism->deskripsi ?? '') ?></textarea>
                </div>
                
                <div>
                    <label><strong>Address:</strong></label>
                    <input type="text" name="alamat_lengkap" class="form-control" value="<?= htmlspecialchars($tourism->alamat ?? '') ?>" required>
                </div>
                
                <div>
                    <label><strong>Adult Ticket Price:</strong></label>
                    <input type="number" name="harga_tiket_dewasa" class="form-control" value="<?= $tourism->harga_tiket_dewasa ?>" required>
                </div>
                
                <div>
                    <label><strong>Child Ticket Price (optional):</strong></label>
                    <input type="number" name="harga_tiket_anak" class="form-control" value="<?= $tourism->harga_tiket_anak ?? '' ?>">
                </div>
                
                <div>
                    <label><strong>Latitude (optional):</strong></label>
                    <input type="number" step="any" name="latitude" class="form-control" value="<?= $tourism->latitude ?? '' ?>">
                </div>
                
                <div>
                    <label><strong>Longitude (optional):</strong></label>
                    <input type="number" step="any" name="longitude" class="form-control" value="<?= $tourism->longitude ?? '' ?>">
                </div>
                
                <div>
                    <label><strong>Main Photo (optional):</strong></label>
                    <input type="file" name="foto_utama" class="form-control" accept="image/*">
                    <small>Current: <?= $tourism->foto_utama ?: 'None' ?></small>
                </div>
                
                <button type="submit" class="btn btn-success">🧪 Test Update Process</button>
            </form>
        </div>
        
        <div class="section">
            <h2>🔗 Admin Panel Links</h2>
            <p><strong>After testing here, try these admin links:</strong></p>
            <ul>
                <li><a href="https://susukannet.id/login" target="_blank">Login to Admin</a></li>
                <li><a href="https://susukannet.id/admin/objek-wisata" target="_blank">Tourism Index</a></li>
                <li><a href="https://susukannet.id/admin/objek-wisata/<?= $tourism->id ?>/edit" target="_blank">Edit Tourism (Real Form)</a></li>
                <li><a href="https://susukannet.id/admin/objek-wisata/<?= $tourism->id ?>" target="_blank">Show Tourism</a></li>
            </ul>
            
            <h3>🔧 Troubleshooting Steps</h3>
            <ol>
                <li><strong>If this test works but admin doesn't:</strong> Authentication issue</li>
                <li><strong>If validation fails:</strong> Check field names and requirements</li>
                <li><strong>If upload fails:</strong> Check file permissions and size</li>
                <li><strong>If redirect to login:</strong> Create/login as admin user</li>
                <li><strong>If 500 error:</strong> Check Laravel logs in storage/logs/</li>
            </ol>
        </div>
    </div>
</body>
</html>
