<?php
/**
 * Test Update Request Simulation
 */

require_once __DIR__ . '/../vendor/autoload.php';

// Load Laravel
$app = require_once __DIR__ . '/../bootstrap/app.php';
$app->make('Illuminate\Contracts\Console\Kernel')->bootstrap();

use App\Models\ObjekWisata;
use App\Http\Controllers\Admin\ObjekWisataController;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

header('Content-Type: text/html; charset=utf-8');

?>
<!DOCTYPE html>
<html>
<head>
    <title>Test Update Request</title>
    <style>
        body { font-family: Arial, sans-serif; margin: 20px; background: #f5f5f5; }
        .container { max-width: 900px; margin: 0 auto; background: white; padding: 20px; border-radius: 8px; box-shadow: 0 2px 10px rgba(0,0,0,0.1); }
        .test-section { margin: 20px 0; padding: 15px; border: 1px solid #ddd; border-radius: 8px; background: #f9f9f9; }
        .success { color: #28a745; background: #d4edda; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .error { color: #dc3545; background: #f8d7da; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .warning { color: #856404; background: #fff3cd; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .info { color: #0c5460; background: #d1ecf1; padding: 8px; border-radius: 4px; margin: 5px 0; }
        .code { background: #f1f1f1; padding: 8px; border-radius: 4px; font-family: monospace; margin: 5px 0; white-space: pre-wrap; }
    </style>
</head>
<body>
    <div class="container">
        <h1>🧪 Test Update Request Simulation</h1>
        
        <?php
        // Test 1: Check current tourism
        echo "<div class='test-section'>";
        echo "<h2>🎯 Test 1: Get Tourism Object</h2>";
        
        try {
            $tourism = ObjekWisata::first();
            
            if ($tourism) {
                echo "<div class='success'>✅ Found tourism: {$tourism->nama_wisata}</div>";
                echo "<div class='info'><strong>ID:</strong> {$tourism->id}</div>";
                echo "<div class='info'><strong>Current photo:</strong> " . ($tourism->foto_utama ?: 'NULL') . "</div>";
                
                $testId = $tourism->id;
            } else {
                echo "<div class='error'>❌ No tourism found</div>";
                exit;
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Error getting tourism: " . $e->getMessage() . "</div>";
            exit;
        }
        echo "</div>";
        
        // Test 2: Test route resolution
        echo "<div class='test-section'>";
        echo "<h2>🔗 Test 2: Route Resolution</h2>";
        
        try {
            $editRoute = route('admin.objek-wisata.edit', $testId);
            $updateRoute = route('admin.objek-wisata.update', $testId);
            
            echo "<div class='info'><strong>Edit Route:</strong> {$editRoute}</div>";
            echo "<div class='info'><strong>Update Route:</strong> {$updateRoute}</div>";
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Route resolution error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Test 3: Simulate controller instantiation
        echo "<div class='test-section'>";
        echo "<h2>🎮 Test 3: Controller Test</h2>";
        
        try {
            $controller = new ObjekWisataController();
            echo "<div class='success'>✅ Controller instantiated</div>";
            
            // Test edit method
            $editResult = $controller->edit($tourism);
            echo "<div class='success'>✅ Edit method executed</div>";
            echo "<div class='info'><strong>Edit result type:</strong> " . get_class($editResult) . "</div>";
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Controller test error: " . $e->getMessage() . "</div>";
            echo "<div class='code'>" . $e->getTraceAsString() . "</div>";
        }
        echo "</div>";
        
        // Test 4: Create mock request
        echo "<div class='test-section'>";
        echo "<h2>📝 Test 4: Mock Update Request</h2>";
        
        try {
            // Create mock request data
            $requestData = [
                'nama_wisata' => 'Test Tourism Updated',
                'kategori_wisata' => 'wisata_alam',
                'deskripsi' => 'Test description for update',
                'alamat_lengkap' => 'Test address updated',
                'harga_tiket_dewasa' => 15000,
                'harga_tiket_anak' => 7500,
                'longitude' => 107.6191,
                'latitude' => -6.9175,
                '_token' => csrf_token(),
                '_method' => 'PUT'
            ];
            
            echo "<div class='info'>📋 Mock request data prepared</div>";
            echo "<div class='code'>" . print_r($requestData, true) . "</div>";
            
            // Create Laravel request object
            $request = new Request();
            $request->replace($requestData);
            $request->setMethod('PUT');
            
            echo "<div class='success'>✅ Laravel Request object created</div>";
            echo "<div class='info'><strong>Method:</strong> " . $request->method() . "</div>";
            echo "<div class='info'><strong>Has nama_wisata:</strong> " . ($request->has('nama_wisata') ? 'YES' : 'NO') . "</div>";
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Mock request error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Test 5: Test validation
        echo "<div class='test-section'>";
        echo "<h2>✅ Test 5: Validation Test</h2>";
        
        try {
            $validator = \Illuminate\Support\Facades\Validator::make($requestData, [
                'nama_wisata' => 'required|string|max:255',
                'kategori_wisata' => 'required|in:wisata_alam,wisata_budaya,wisata_sejarah,wisata_religi,wisata_kuliner,wisata_edukasi,wisata_adventure,wisata_air,wisata_pegunungan,agrowisata,ekowisata,wisata_buatan,lainnya',
                'deskripsi' => 'required|string',
                'alamat_lengkap' => 'required|string',
                'harga_tiket_dewasa' => 'required|numeric|min:0',
                'foto_utama' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
                'galeri_foto.*' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
            ]);
            
            if ($validator->fails()) {
                echo "<div class='error'>❌ Validation failed:</div>";
                foreach ($validator->errors()->all() as $error) {
                    echo "<div class='error'>• {$error}</div>";
                }
            } else {
                echo "<div class='success'>✅ Validation passed</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Validation test error: " . $e->getMessage() . "</div>";
        }
        echo "</div>";
        
        // Test 6: Test actual update (without file)
        echo "<div class='test-section'>";
        echo "<h2>🔄 Test 6: Actual Update Test</h2>";
        
        try {
            $oldData = [
                'nama_wisata' => $tourism->nama_wisata,
                'kategori_wisata' => $tourism->kategori_wisata,
                'deskripsi' => $tourism->deskripsi
            ];
            
            echo "<div class='info'><strong>Before update:</strong></div>";
            echo "<div class='code'>" . print_r($oldData, true) . "</div>";
            
            // Update tourism directly (without controller)
            $updateData = [
                'nama_wisata' => 'Test Updated - ' . date('H:i:s'),
                'kategori_wisata' => 'wisata_alam',
                'deskripsi' => 'Updated description at ' . date('Y-m-d H:i:s'),
                'alamat' => 'Updated address',
                'harga_tiket_dewasa' => 20000
            ];
            
            $result = $tourism->update($updateData);
            
            if ($result) {
                echo "<div class='success'>✅ Direct update successful</div>";
                
                $updatedTourism = $tourism->fresh();
                echo "<div class='info'><strong>After update:</strong></div>";
                echo "<div class='code'>";
                echo "nama_wisata: {$updatedTourism->nama_wisata}\n";
                echo "kategori_wisata: {$updatedTourism->kategori_wisata}\n";
                echo "deskripsi: {$updatedTourism->deskripsi}\n";
                echo "alamat: {$updatedTourism->alamat}\n";
                echo "harga_tiket_dewasa: {$updatedTourism->harga_tiket_dewasa}";
                echo "</div>";
                
            } else {
                echo "<div class='error'>❌ Direct update failed</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Update test error: " . $e->getMessage() . "</div>";
            echo "<div class='code'>" . $e->getTraceAsString() . "</div>";
        }
        echo "</div>";
        
        // Test 7: Test with controller update method
        echo "<div class='test-section'>";
        echo "<h2>🎮 Test 7: Controller Update Method</h2>";
        
        try {
            // Simulate authenticated user (admin)
            $adminUser = \App\Models\User::where('role', 'admin')->first();
            
            if ($adminUser) {
                Auth::login($adminUser);
                echo "<div class='success'>✅ Authenticated as admin: {$adminUser->name}</div>";
                
                // Try controller update method
                $controller = new ObjekWisataController();
                
                // Create proper request
                $request = Request::create(
                    route('admin.objek-wisata.update', $tourism->id),
                    'PUT',
                    [
                        'nama_wisata' => 'Controller Test - ' . date('H:i:s'),
                        'kategori_wisata' => 'wisata_budaya',
                        'deskripsi' => 'Controller test description',
                        'alamat_lengkap' => 'Controller test address',
                        'harga_tiket_dewasa' => 25000,
                        '_token' => csrf_token()
                    ]
                );
                
                echo "<div class='info'>📝 Created proper update request</div>";
                
                // Call controller update method
                $response = $controller->update($request, $tourism);
                
                echo "<div class='success'>✅ Controller update method executed</div>";
                echo "<div class='info'><strong>Response type:</strong> " . get_class($response) . "</div>";
                
                if (method_exists($response, 'getStatusCode')) {
                    echo "<div class='info'><strong>Status code:</strong> " . $response->getStatusCode() . "</div>";
                }
                
                // Check if tourism was updated
                $finalTourism = $tourism->fresh();
                echo "<div class='info'><strong>Final name:</strong> {$finalTourism->nama_wisata}</div>";
                
            } else {
                echo "<div class='warning'>⚠️ No admin user found for authentication test</div>";
            }
            
        } catch (Exception $e) {
            echo "<div class='error'>❌ Controller update error: " . $e->getMessage() . "</div>";
            echo "<div class='code'>" . $e->getTraceAsString() . "</div>";
        }
        echo "</div>";
        ?>
        
        <div class="test-section">
            <h2>📋 Test Results Summary</h2>
            <p><strong>If all tests pass but the form still doesn't work:</strong></p>
            <ol>
                <li><strong>Check browser network tab</strong> for HTTP request/response details</li>
                <li><strong>Check CSRF token</strong> in the form</li>
                <li><strong>Check authentication</strong> - make sure you're logged in as admin</li>
                <li><strong>Check form enctype</strong> - must be multipart/form-data</li>
                <li><strong>Check file upload</strong> - try without image first</li>
                <li><strong>Check Laravel logs</strong> for detailed error messages</li>
            </ol>
            
            <h3>🔗 Test in Browser</h3>
            <ul>
                <li><a href="https://susukannet.id/debug-edit-update.php" target="_blank">Debug Edit Update</a></li>
                <li><a href="https://susukannet.id/check-errors.php" target="_blank">Check Errors</a></li>
                <li><a href="https://susukannet.id/admin/objek-wisata/<?= $testId ?>/edit" target="_blank">Edit Form</a></li>
            </ul>
        </div>
    </div>
</body>
</html>
