@extends('layouts.app')

@section('title', 'Bug Analytics Dashboard')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Dashboard</a></li>
    <li class="breadcrumb-item"><a href="{{ route('admin.bugs.index') }}">Bug Detection</a></li>
    <li class="breadcrumb-item active">Analytics</li>
@endsection

@section('content')
<!-- Page Header -->
<div class="d-flex justify-content-between align-items-center mb-4">
    <div>
        <h2 class="mb-2">
            <i class="fas fa-chart-line me-2 text-info"></i>
            Bug Analytics Dashboard
        </h2>
        <p class="text-muted mb-0">Analisis mendalam tentang bug dan error dalam sistem</p>
    </div>
    <div>
        <a href="{{ route('admin.bugs.index') }}" class="btn btn-outline-primary me-2">
            <i class="fas fa-list me-2"></i>
            Bug List
        </a>
        <button class="btn btn-gradient-info" onclick="runHealthCheck()">
            <i class="fas fa-heartbeat me-2"></i>
            Health Check
        </button>
    </div>
</div>

<!-- Quick Stats -->
<div class="row mb-4">
    <div class="col-md-3">
        <div class="stats-card bg-gradient-danger text-white">
            <div class="d-flex align-items-center">
                <div class="stats-icon me-3">
                    <i class="fas fa-exclamation-triangle"></i>
                </div>
                <div>
                    <h3 class="mb-0 text-white">{{ $stats['total'] }}</h3>
                    <small>Total Bugs</small>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stats-card bg-gradient-warning text-white">
            <div class="d-flex align-items-center">
                <div class="stats-icon me-3">
                    <i class="fas fa-folder-open"></i>
                </div>
                <div>
                    <h3 class="mb-0 text-white">{{ $stats['open'] }}</h3>
                    <small>Open Bugs</small>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stats-card bg-gradient-dark text-white">
            <div class="d-flex align-items-center">
                <div class="stats-icon me-3">
                    <i class="fas fa-skull"></i>
                </div>
                <div>
                    <h3 class="mb-0 text-white">{{ $stats['critical'] }}</h3>
                    <small>Critical Bugs</small>
                </div>
            </div>
        </div>
    </div>
    <div class="col-md-3">
        <div class="stats-card bg-gradient-success text-white">
            <div class="d-flex align-items-center">
                <div class="stats-icon me-3">
                    <i class="fas fa-percentage"></i>
                </div>
                <div>
                    <h3 class="mb-0 text-white">{{ $stats['resolution_rate'] ?? 0 }}%</h3>
                    <small>Resolution Rate</small>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Charts Row -->
<div class="row mb-4">
    <!-- Bug Trends Chart -->
    <div class="col-lg-8">
        <div class="card">
            <div class="card-header bg-light">
                <h5 class="card-title mb-0">
                    <i class="fas fa-chart-line me-2"></i>
                    Bug Trends (Last 7 Days)
                </h5>
            </div>
            <div class="card-body">
                <canvas id="bugTrendsChart" height="100"></canvas>
            </div>
        </div>
    </div>
    
    <!-- Severity Distribution -->
    <div class="col-lg-4">
        <div class="card">
            <div class="card-header bg-light">
                <h5 class="card-title mb-0">
                    <i class="fas fa-chart-pie me-2"></i>
                    Severity Distribution
                </h5>
            </div>
            <div class="card-body">
                <canvas id="severityChart" height="200"></canvas>
            </div>
        </div>
    </div>
</div>

<!-- Critical Bugs & Top Modules -->
<div class="row mb-4">
    <!-- Critical Bugs -->
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header bg-danger text-white">
                <h5 class="card-title mb-0">
                    <i class="fas fa-fire me-2"></i>
                    Critical Bugs ({{ $criticalBugs->count() }})
                </h5>
            </div>
            <div class="card-body">
                @if($criticalBugs->count() > 0)
                    <div class="list-group list-group-flush">
                        @foreach($criticalBugs as $bug)
                        <div class="list-group-item d-flex justify-content-between align-items-start">
                            <div class="ms-2 me-auto">
                                <div class="fw-bold">{{ Str::limit($bug->title, 40) }}</div>
                                <small class="text-muted">
                                    {{ $bug->module ?? 'Unknown' }} • 
                                    {{ $bug->created_at->diffForHumans() }}
                                </small>
                            </div>
                            <div>
                                <span class="badge bg-danger">{{ $bug->occurrence_count }}</span>
                                <a href="{{ route('admin.bugs.show', $bug) }}" class="btn btn-sm btn-outline-primary ms-2">
                                    <i class="fas fa-eye"></i>
                                </a>
                            </div>
                        </div>
                        @endforeach
                    </div>
                @else
                    <div class="text-center py-4">
                        <i class="fas fa-check-circle fa-3x text-success mb-3"></i>
                        <h6 class="text-muted">No Critical Bugs</h6>
                        <p class="text-muted">Great! No critical bugs found.</p>
                    </div>
                @endif
            </div>
        </div>
    </div>
    
    <!-- Top Error Modules -->
    <div class="col-lg-6">
        <div class="card">
            <div class="card-header bg-warning text-dark">
                <h5 class="card-title mb-0">
                    <i class="fas fa-cubes me-2"></i>
                    Top Error Modules
                </h5>
            </div>
            <div class="card-body">
                @if($topModules->count() > 0)
                    <div class="list-group list-group-flush">
                        @foreach($topModules as $module)
                        <div class="list-group-item d-flex justify-content-between align-items-center">
                            <div>
                                <div class="fw-bold">{{ $module->module ?? 'Unknown' }}</div>
                                <small class="text-muted">Last month</small>
                            </div>
                            <div class="d-flex align-items-center">
                                <div class="progress me-3" style="width: 100px; height: 8px;">
                                    <div class="progress-bar bg-warning" 
                                         style="width: {{ min(($module->bug_count / $topModules->first()->bug_count) * 100, 100) }}%"></div>
                                </div>
                                <span class="badge bg-warning text-dark">{{ $module->bug_count }}</span>
                            </div>
                        </div>
                        @endforeach
                    </div>
                @else
                    <div class="text-center py-4">
                        <i class="fas fa-cube fa-3x text-muted mb-3"></i>
                        <h6 class="text-muted">No Module Data</h6>
                        <p class="text-muted">No module-specific errors found.</p>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<!-- System Health Status -->
<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header bg-info text-white">
                <h5 class="card-title mb-0">
                    <i class="fas fa-heartbeat me-2"></i>
                    System Health Status
                </h5>
            </div>
            <div class="card-body" id="healthStatus">
                <div class="text-center py-4">
                    <div class="spinner-border text-info" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2 text-muted">Checking system health...</p>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Recent Activity -->
<div class="row">
    <div class="col-12">
        <div class="card">
            <div class="card-header bg-light">
                <h5 class="card-title mb-0">
                    <i class="fas fa-history me-2"></i>
                    Recent Bug Activity
                </h5>
            </div>
            <div class="card-body">
                <div class="table-responsive">
                    <table class="table table-hover">
                        <thead class="table-light">
                            <tr>
                                <th>Time</th>
                                <th>Bug</th>
                                <th>Severity</th>
                                <th>Module</th>
                                <th>Status</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @php
                                $recentBugs = \App\Models\Bug::with(['user', 'resolver'])
                                                ->latest()
                                                ->limit(10)
                                                ->get();
                            @endphp
                            @forelse($recentBugs as $bug)
                            <tr>
                                <td>
                                    <small class="text-muted">{{ $bug->created_at->diffForHumans() }}</small>
                                </td>
                                <td>
                                    <div class="fw-semibold">{{ Str::limit($bug->title, 30) }}</div>
                                    <small class="text-muted">{{ Str::limit($bug->description, 40) }}</small>
                                </td>
                                <td>
                                    <span class="badge bg-{{ $bug->severity_badge_color }}">
                                        {{ ucfirst($bug->severity) }}
                                    </span>
                                </td>
                                <td>
                                    <span class="text-muted">{{ $bug->module ?? 'N/A' }}</span>
                                </td>
                                <td>
                                    <span class="badge bg-{{ $bug->status_badge_color }}">
                                        {{ str_replace('_', ' ', ucfirst($bug->status)) }}
                                    </span>
                                </td>
                                <td>
                                    <a href="{{ route('admin.bugs.show', $bug) }}" class="btn btn-sm btn-outline-primary">
                                        <i class="fas fa-eye"></i>
                                    </a>
                                </td>
                            </tr>
                            @empty
                            <tr>
                                <td colspan="6" class="text-center py-4">
                                    <i class="fas fa-inbox fa-2x text-muted mb-2"></i>
                                    <p class="text-muted">No recent bug activity</p>
                                </td>
                            </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>
</div>

@endsection

@push('styles')
<style>
.stats-card {
    background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
    border-radius: 15px;
    padding: 20px;
    margin-bottom: 20px;
    box-shadow: 0 10px 20px rgba(0,0,0,0.1);
    transition: transform 0.3s ease;
}

.stats-card:hover {
    transform: translateY(-5px);
}

.stats-icon {
    width: 50px;
    height: 50px;
    background: rgba(255,255,255,0.2);
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    font-size: 1.2rem;
}

.bg-gradient-danger {
    background: linear-gradient(135deg, #ff416c 0%, #ff4757 100%);
}

.bg-gradient-warning {
    background: linear-gradient(135deg, #f7b731 0%, #f39801 100%);
}

.bg-gradient-success {
    background: linear-gradient(135deg, #2ed573 0%, #1dd1a1 100%);
}

.bg-gradient-info {
    background: linear-gradient(135deg, #3742fa 0%, #2f3542 100%);
}

.bg-gradient-dark {
    background: linear-gradient(135deg, #2c2c54 0%, #40407a 100%);
}

.btn-gradient-info {
    background: linear-gradient(135deg, #3742fa 0%, #2f3542 100%);
    border: none;
    color: white;
    transition: all 0.3s ease;
}

.btn-gradient-info:hover {
    background: linear-gradient(135deg, #2f3542 0%, #3742fa 100%);
    color: white;
    transform: translateY(-2px);
    box-shadow: 0 4px 15px rgba(55,66,250,0.3);
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
// Bug Trends Chart
const bugTrendsCtx = document.getElementById('bugTrendsChart').getContext('2d');
const bugTrendsChart = new Chart(bugTrendsCtx, {
    type: 'line',
    data: {
        labels: {!! json_encode(array_keys($bugsByDay)) !!},
        datasets: [{
            label: 'New Bugs',
            data: {!! json_encode(array_values($bugsByDay)) !!},
            borderColor: '#ff416c',
            backgroundColor: 'rgba(255, 65, 108, 0.1)',
            borderWidth: 3,
            fill: true,
            tension: 0.4
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                display: false
            }
        },
        scales: {
            y: {
                beginAtZero: true,
                ticks: {
                    stepSize: 1
                }
            }
        }
    }
});

// Severity Distribution Chart
const severityCtx = document.getElementById('severityChart').getContext('2d');
const severityData = {!! json_encode($stats['by_severity'] ?? []) !!};
const severityChart = new Chart(severityCtx, {
    type: 'doughnut',
    data: {
        labels: ['Critical', 'High', 'Medium', 'Low'],
        datasets: [{
            data: [
                severityData.critical || 0,
                severityData.high || 0,
                severityData.medium || 0,
                severityData.low || 0
            ],
            backgroundColor: [
                '#2c2c54',
                '#ff4757',
                '#f39801',
                '#2ed573'
            ],
            borderWidth: 0
        }]
    },
    options: {
        responsive: true,
        maintainAspectRatio: false,
        plugins: {
            legend: {
                position: 'bottom'
            }
        }
    }
});

// Health Check Function
function runHealthCheck() {
    const healthStatus = document.getElementById('healthStatus');
    
    healthStatus.innerHTML = `
        <div class="text-center py-4">
            <div class="spinner-border text-info" role="status">
                <span class="visually-hidden">Loading...</span>
            </div>
            <p class="mt-2 text-muted">Running comprehensive health check...</p>
        </div>
    `;
    
    fetch('/admin/bugs/health-check')
        .then(response => response.json())
        .then(data => {
            let html = '';
            
            if (data.status === 'healthy') {
                html = `
                    <div class="alert alert-success">
                        <div class="d-flex align-items-center">
                            <i class="fas fa-check-circle fa-2x me-3"></i>
                            <div>
                                <h5 class="mb-1">System Healthy</h5>
                                <p class="mb-0">All systems are running normally. No issues detected.</p>
                            </div>
                        </div>
                    </div>
                `;
            } else {
                html = `
                    <div class="alert alert-warning">
                        <div class="d-flex align-items-center mb-3">
                            <i class="fas fa-exclamation-triangle fa-2x me-3"></i>
                            <div>
                                <h5 class="mb-1">Issues Detected</h5>
                                <p class="mb-0">${data.issues.length} issue(s) found that need attention.</p>
                            </div>
                        </div>
                        <div class="row">
                `;
                
                data.issues.forEach(issue => {
                    const iconClass = issue.type === 'critical' ? 'fa-exclamation-circle text-danger' :
                                    issue.type === 'high' ? 'fa-exclamation-triangle text-warning' :
                                    'fa-info-circle text-info';
                    
                    html += `
                        <div class="col-md-6 mb-2">
                            <div class="d-flex align-items-start">
                                <i class="fas ${iconClass} me-2 mt-1"></i>
                                <div>
                                    <strong>${issue.module}</strong><br>
                                    <small class="text-muted">${issue.message}</small>
                                </div>
                            </div>
                        </div>
                    `;
                });
                
                html += `
                        </div>
                    </div>
                `;
            }
            
            healthStatus.innerHTML = html;
        })
        .catch(error => {
            healthStatus.innerHTML = `
                <div class="alert alert-danger">
                    <div class="d-flex align-items-center">
                        <i class="fas fa-times-circle fa-2x me-3"></i>
                        <div>
                            <h5 class="mb-1">Health Check Failed</h5>
                            <p class="mb-0">Unable to perform health check: ${error.message}</p>
                        </div>
                    </div>
                </div>
            `;
        });
}

// Auto-run health check on page load
document.addEventListener('DOMContentLoaded', function() {
    setTimeout(runHealthCheck, 1000);
});

// Auto-refresh charts every 30 seconds
setInterval(() => {
    fetch('/admin/bugs/statistics')
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                // Update charts with new data
                console.log('Charts updated with latest data');
            }
        })
        .catch(error => console.log('Auto-refresh failed:', error));
}, 30000);
</script>
@endpush 