@extends('layouts.app')

@section('title', 'Detail Penduduk')

@section('breadcrumb')
    <li class="breadcrumb-item"><a href="{{ route('admin.dashboard') }}">Dashboard</a></li>
    <li class="breadcrumb-item"><a href="{{ route('admin.penduduk.index') }}">Data Penduduk</a></li>
    <li class="breadcrumb-item active">{{ $penduduk->nama }}</li>
@endsection

@section('content')
<!-- Page Header -->
<div class="page-header mb-4">
    <div class="d-flex justify-content-between align-items-start">
        <div class="page-title-section">
            <div class="d-flex align-items-center mb-2">
                <div class="page-icon bg-primary text-white rounded-circle me-3">
                    <i class="fas fa-user"></i>
                </div>
                <div>
                    <h1 class="page-title mb-1">{{ $penduduk->nama }}</h1>
                    <p class="page-subtitle text-muted mb-0">
                        <i class="fas fa-id-card me-1"></i>{{ $penduduk->nik }}
                        @if($penduduk->status_hidup !== 'Hidup')
                            <span class="badge bg-danger ms-2">{{ $penduduk->status_hidup }}</span>
                        @endif
                    </p>
                </div>
            </div>
        </div>
        <div class="page-actions">
            <div class="btn-toolbar" role="toolbar">
                <div class="btn-group me-2" role="group">
                    <a href="{{ route('admin.penduduk.index') }}" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left me-1"></i>
                        <span class="d-none d-sm-inline">Kembali</span>
                    </a>
                    <a href="{{ route('admin.penduduk.edit', $penduduk->id) }}" class="btn btn-warning">
                        <i class="fas fa-edit me-1"></i>
                        <span class="d-none d-sm-inline">Edit</span>
                    </a>
                </div>
                <div class="btn-group" role="group">
                    <button type="button" class="btn btn-primary dropdown-toggle" data-bs-toggle="dropdown">
                        <i class="fas fa-print me-1"></i>
                        <span class="d-none d-sm-inline">Cetak</span>
                    </button>
                    <ul class="dropdown-menu dropdown-menu-end">
                        <li><a class="dropdown-item" href="#" onclick="printKTP()">
                            <i class="fas fa-id-card me-2"></i>KTP
                        </a></li>
                        <li><a class="dropdown-item" href="#" onclick="printKK()">
                            <i class="fas fa-users me-2"></i>Kartu Keluarga
                        </a></li>
                        <li><a class="dropdown-item" href="#" onclick="printProfile()">
                            <i class="fas fa-user me-2"></i>Profil Lengkap
                        </a></li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <!-- Main Profile -->
    <div class="col-lg-8">
        <!-- Personal Information -->
        <div class="card profile-card shadow-sm">
            <div class="card-body p-4">
                <div class="row">
                    <!-- Photo Section -->
                    <div class="col-lg-3 col-md-4 text-center">
                        <div class="profile-photo-section">
                            <div class="profile-photo-container mb-3">
                                @if($penduduk->foto)
                                    <img src="{{ \App\Helpers\ImageHelper::getAvatarUrl($penduduk->foto, $penduduk->nama) }}" 
                                         class="profile-photo"
                                         alt="Foto {{ $penduduk->nama }}"
                                         onclick="showPhotoModal('{{ \App\Helpers\ImageHelper::getAvatarUrl($penduduk->foto, $penduduk->nama) }}')">
                                @else
                                    <div class="profile-photo-placeholder">
                                        <div class="avatar-initials">
                                            {{ strtoupper(substr($penduduk->nama, 0, 2)) }}
                                        </div>
                                    </div>
                                @endif
                                <div class="profile-status-badge">
                                    <span class="badge bg-{{ $penduduk->status_hidup === 'Hidup' ? 'success' : 'danger' }}">
                                        {{ $penduduk->status_hidup ?? 'Hidup' }}
                                    </span>
                                </div>
                            </div>
                            
                            <!-- Profile Summary -->
                            <div class="profile-summary mb-4">
                                <h4 class="profile-name mb-1">{{ $penduduk->nama }}</h4>
                                <p class="profile-nik text-muted mb-2">{{ $penduduk->nik }}</p>
                                <div class="profile-badges mb-3">
                                    <span class="badge bg-{{ $penduduk->jenis_kelamin === 'L' ? 'primary' : 'pink' }} me-1">
                                        {{ $penduduk->jenis_kelamin === 'L' ? 'Laki-laki' : 'Perempuan' }}
                                    </span>
                                    @if($penduduk->tanggal_lahir)
                                        <span class="badge bg-info">{{ $penduduk->tanggal_lahir->age }} tahun</span>
                                    @endif
                                </div>
                            </div>
                            
                            <!-- Quick Contact Actions -->
                            <div class="profile-actions d-grid gap-2">
                                @if($penduduk->no_wa)
                                    <button type="button" class="btn btn-success btn-sm" onclick="sendWhatsApp()">
                                        <i class="fab fa-whatsapp me-2"></i>
                                        WhatsApp
                                    </button>
                                @endif
                                @if($penduduk->latitude && $penduduk->longitude)
                                    <button type="button" class="btn btn-info btn-sm" onclick="showOnMap()">
                                        <i class="fas fa-map-marker-alt me-2"></i>
                                        Lihat di Peta
                                    </button>
                                @endif
                                <button type="button" class="btn btn-outline-primary btn-sm" onclick="createSurat()">
                                    <i class="fas fa-file-plus me-2"></i>
                                    Buat Surat
                                </button>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Personal Details -->
                    <div class="col-lg-9 col-md-8">
                        <div class="profile-details">
                            <div class="section-header mb-4">
                                <h5 class="section-title">
                                    <i class="fas fa-id-card me-2 text-primary"></i>
                                    Informasi Pribadi
                                </h5>
                            </div>
                            
                            <div class="row g-4">
                                <!-- Identitas -->
                                <div class="col-md-6">
                                    <div class="info-group">
                                        <h6 class="info-group-title">Identitas</h6>
                                        <div class="info-list">
                                            <div class="info-item">
                                                <span class="info-label">NIK</span>
                                                <span class="info-value">{{ $penduduk->nik }}</span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">No. KK</span>
                                                <span class="info-value">
                                                    @if($penduduk->keluarga)
                                                        <a href="{{ route('admin.keluarga.show', $penduduk->keluarga->id) }}" 
                                                           class="text-primary text-decoration-none">
                                                            {{ $penduduk->no_kk }}
                                                            <i class="fas fa-external-link-alt fa-xs ms-1"></i>
                                                        </a>
                                                    @else
                                                        {{ $penduduk->no_kk ?? '-' }}
                                                    @endif
                                                </span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Tempat, Tanggal Lahir</span>
                                                <span class="info-value">
                                                    {{ $penduduk->tempat_lahir ?? '-' }}
                                                    @if($penduduk->tanggal_lahir)
                                                        , {{ $penduduk->tanggal_lahir->format('d F Y') }}
                                                    @endif
                                                </span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Agama</span>
                                                <span class="info-value">{{ $penduduk->agama ?? '-' }}</span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Kewarganegaraan</span>
                                                <span class="info-value">{{ $penduduk->kewarganegaraan ?? 'WNI' }}</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Data Sosial -->
                                <div class="col-md-6">
                                    <div class="info-group">
                                        <h6 class="info-group-title">Data Sosial</h6>
                                        <div class="info-list">
                                            <div class="info-item">
                                                <span class="info-label">Pendidikan Terakhir</span>
                                                <span class="info-value">{{ $penduduk->pendidikan_terakhir ?? '-' }}</span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Pekerjaan</span>
                                                <span class="info-value">{{ $penduduk->pekerjaan ?? '-' }}</span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Status Perkawinan</span>
                                                <span class="info-value">{{ $penduduk->status_kawin ?? 'Belum Kawin' }}</span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Golongan Darah</span>
                                                <span class="info-value">
                                                    @if($penduduk->golongan_darah)
                                                        <span class="badge bg-danger">{{ $penduduk->golongan_darah }}</span>
                                                    @else
                                                        -
                                                    @endif
                                                </span>
                                            </div>
                                            <div class="info-item">
                                                <span class="info-label">Status Keluarga</span>
                                                <span class="info-value">{{ $penduduk->status_keluarga ?? '-' }}</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                                
                                <!-- Kontak -->
                                <div class="col-12">
                                    <div class="info-group">
                                        <h6 class="info-group-title">Informasi Kontak</h6>
                                        <div class="row">
                                            <div class="col-md-6">
                                                <div class="info-item">
                                                    <span class="info-label">No. WhatsApp</span>
                                                    <span class="info-value">
                                                        @if($penduduk->no_wa)
                                                            <a href="https://wa.me/{{ $penduduk->no_wa }}" target="_blank" 
                                                               class="text-success text-decoration-none">
                                                                <i class="fab fa-whatsapp me-1"></i>{{ $penduduk->no_wa }}
                                                            </a>
                                                        @else
                                                            -
                                                        @endif
                                                    </span>
                                                </div>
                                            </div>
                                            <div class="col-md-6">
                                                <div class="info-item">
                                                    <span class="info-label">Email</span>
                                                    <span class="info-value">
                                                        @if($penduduk->email)
                                                            <a href="mailto:{{ $penduduk->email }}" class="text-primary text-decoration-none">
                                                                <i class="fas fa-envelope me-1"></i>{{ $penduduk->email }}
                                                            </a>
                                                        @else
                                                            -
                                                        @endif
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Address Information -->
        <div class="card mt-4 shadow-sm">
            <div class="card-body p-4">
                <div class="section-header mb-4">
                    <h5 class="section-title">
                        <i class="fas fa-map-marker-alt me-2 text-primary"></i>
                        Informasi Alamat
                    </h5>
                </div>
                
                <div class="row g-4">
                    <div class="col-lg-6">
                        <div class="address-details">
                            <div class="info-list">
                                <div class="info-item">
                                    <span class="info-label">Alamat Lengkap</span>
                                    <span class="info-value">{{ $penduduk->alamat ?? '-' }}</span>
                                </div>
                                <div class="row">
                                    <div class="col-4">
                                        <div class="info-item">
                                            <span class="info-label">RT</span>
                                            <span class="info-value">{{ $penduduk->rt ?? '-' }}</span>
                                        </div>
                                    </div>
                                    <div class="col-4">
                                        <div class="info-item">
                                            <span class="info-label">RW</span>
                                            <span class="info-value">{{ $penduduk->rw ?? '-' }}</span>
                                        </div>
                                    </div>
                                    <div class="col-4">
                                        <div class="info-item">
                                            <span class="info-label">Dusun</span>
                                            <span class="info-value">{{ $penduduk->dusun ?? '-' }}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div class="col-lg-6">
                        @if($penduduk->latitude && $penduduk->longitude)
                        <div class="location-info">
                            <div class="coordinates-card bg-light rounded p-3">
                                <div class="d-flex align-items-center justify-content-between mb-3">
                                    <h6 class="mb-0">
                                        <i class="fas fa-crosshairs me-2 text-info"></i>
                                        Koordinat GPS
                                    </h6>
                                    <span class="badge bg-success">Tersedia</span>
                                </div>
                                <div class="row text-center mb-3">
                                    <div class="col-6">
                                        <div class="coordinate-item">
                                            <div class="coordinate-value">{{ $penduduk->latitude }}</div>
                                            <small class="text-muted">Latitude</small>
                                        </div>
                                    </div>
                                    <div class="col-6">
                                        <div class="coordinate-item">
                                            <div class="coordinate-value">{{ $penduduk->longitude }}</div>
                                            <small class="text-muted">Longitude</small>
                                        </div>
                                    </div>
                                </div>
                                <div class="d-grid gap-2">
                                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="openInGoogleMaps()">
                                        <i class="fab fa-google me-2"></i>
                                        Buka di Google Maps
                                    </button>
                                    <button type="button" class="btn btn-outline-info btn-sm" onclick="getDirections()">
                                        <i class="fas fa-route me-2"></i>
                                        Petunjuk Arah
                                    </button>
                                </div>
                            </div>
                        </div>
                        @else
                        <div class="location-placeholder text-center p-4">
                            <div class="placeholder-icon mb-3">
                                <i class="fas fa-map-marker-alt fa-3x text-muted"></i>
                            </div>
                            <h6 class="text-muted mb-2">Koordinat Belum Diset</h6>
                            <p class="text-muted small mb-3">Koordinat GPS rumah belum tersedia</p>
                            <a href="{{ route('admin.penduduk.edit', $penduduk->id) }}" class="btn btn-outline-primary btn-sm">
                                <i class="fas fa-map-pin me-2"></i>
                                Set Koordinat
                            </a>
                        </div>
                        @endif
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Family Information -->
        @if($penduduk->keluarga)
        <div class="card mt-4 shadow-sm">
            <div class="card-body p-4">
                <div class="section-header mb-4">
                    <h5 class="section-title">
                        <i class="fas fa-users me-2 text-primary"></i>
                        Informasi Keluarga
                    </h5>
                </div>
                
                <div class="family-info">
                    <!-- Family Header -->
                    <div class="family-header bg-light rounded p-3 mb-4">
                        <div class="d-flex justify-content-between align-items-center">
                            <div class="family-head-info">
                                <h6 class="mb-1">
                                    <i class="fas fa-crown text-warning me-2"></i>
                                    Kepala Keluarga: {{ $penduduk->keluarga->kepala_keluarga->nama ?? '-' }}
                                </h6>
                                <p class="text-muted mb-0">
                                    <i class="fas fa-id-card me-1"></i>
                                    No. KK: {{ $penduduk->keluarga->no_kk }}
                                </p>
                            </div>
                            <a href="{{ route('admin.keluarga.show', $penduduk->keluarga->id) }}" 
                               class="btn btn-outline-primary btn-sm">
                                <i class="fas fa-external-link-alt me-1"></i>
                                Detail Keluarga
                            </a>
                        </div>
                    </div>
                    
                    <!-- Family Members -->
                    <div class="family-members">
                        <h6 class="mb-3">
                            <i class="fas fa-users me-2"></i>
                            Anggota Keluarga ({{ $penduduk->keluarga->anggota->count() }} orang)
                        </h6>
                        <div class="row g-3">
                            @foreach($penduduk->keluarga->anggota as $anggota)
                            <div class="col-lg-6">
                                <div class="member-card p-3 rounded {{ $anggota->id === $penduduk->id ? 'border-primary bg-primary bg-opacity-10' : '' }}">
                                    <div class="d-flex align-items-center">
                                        <div class="member-avatar bg-{{ $anggota->jenis_kelamin === 'L' ? 'primary' : 'pink' }} text-white rounded-circle me-3" 
                                             style="width: 45px; height: 45px;">
                                            <i class="fas fa-user"></i>
                                        </div>
                                        <div class="flex-grow-1">
                                            <div class="member-name fw-semibold">{{ $anggota->nama }}</div>
                                            <div class="member-relation text-muted small">
                                                {{ $anggota->status_keluarga ?? 'Anggota Keluarga' }}
                                            </div>
                                            <div class="member-details text-muted small">
                                                {{ $anggota->jenis_kelamin === 'L' ? 'Laki-laki' : 'Perempuan' }}
                                                @if($anggota->tanggal_lahir)
                                                    • {{ $anggota->tanggal_lahir->age }} tahun
                                                @endif
                                            </div>
                                        </div>
                                        <div class="member-actions">
                                            @if($anggota->id === $penduduk->id)
                                                <span class="badge bg-primary">
                                                    <i class="fas fa-star me-1"></i>Anda
                                                </span>
                                            @else
                                                <a href="{{ route('admin.penduduk.show', $anggota->id) }}" 
                                                   class="btn btn-outline-info btn-sm">
                                                    <i class="fas fa-eye"></i>
                                                </a>
                                            @endif
                                        </div>
                                    </div>
                                </div>
                            </div>
                            @endforeach
                        </div>
                    </div>
                </div>
            </div>
        </div>
        @else
        <div class="card mt-4 shadow-sm">
            <div class="card-body p-4 text-center">
                <div class="empty-state py-4">
                    <i class="fas fa-users fa-3x text-muted mb-3"></i>
                    <h6 class="text-muted mb-2">Belum Terdaftar dalam Keluarga</h6>
                    <p class="text-muted small mb-3">Data keluarga belum tersedia atau belum terhubung</p>
                    <a href="{{ route('admin.penduduk.edit', $penduduk->id) }}" class="btn btn-outline-primary btn-sm">
                        <i class="fas fa-link me-2"></i>
                        Hubungkan ke Keluarga
                    </a>
                </div>
            </div>
        </div>
        @endif
        
        <!-- Services History -->
        <div class="card mt-4">
            <div class="card-header">
                <h5 class="card-title mb-0">
                    <i class="fas fa-file-alt me-2"></i>
                    Riwayat Pelayanan
                </h5>
            </div>
            <div class="card-body">
                @if(isset($services) && $services->count() > 0)
                <div class="services-timeline">
                    @foreach($services as $service)
                    <div class="service-item d-flex mb-3">
                        <div class="service-icon bg-{{ $service->status_color }} rounded-circle me-3" style="width: 40px; height: 40px;">
                            <i class="fas fa-{{ $service->type_icon }}"></i>
                        </div>
                        <div class="service-content">
                            <h6 class="mb-1">{{ $service->jenis_layanan }}</h6>
                            <p class="mb-1 text-muted">{{ $service->keterangan }}</p>
                            <div class="d-flex align-items-center gap-3">
                                <small class="text-muted">
                                    <i class="fas fa-calendar me-1"></i>
                                    {{ $service->created_at->format('d M Y') }}
                                </small>
                                <span class="badge bg-{{ $service->status_color }}">{{ $service->status }}</span>
                            </div>
                        </div>
                    </div>
                    @endforeach
                </div>
                @else
                <div class="text-center py-4">
                    <i class="fas fa-file-alt fa-3x text-muted mb-3"></i>
                    <h6 class="text-muted">Belum ada riwayat pelayanan</h6>
                    <p class="text-muted">Riwayat surat dan layanan akan muncul di sini</p>
                </div>
                @endif
            </div>
        </div>
    </div>
    
    <!-- Sidebar -->
    <div class="col-lg-4">
        <!-- Quick Stats -->
        <div class="card">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-chart-bar me-2"></i>
                    Statistik
                </h6>
            </div>
            <div class="card-body">
                <div class="stats-grid">
                    <div class="stat-item text-center p-3 bg-primary bg-opacity-10 rounded mb-3">
                        <h4 class="mb-1 text-primary">{{ $penduduk->umur }}</h4>
                        <small class="text-muted">Tahun</small>
                    </div>
                    <div class="row">
                        <div class="col-6">
                            <div class="stat-item text-center p-2 bg-success bg-opacity-10 rounded">
                                <h5 class="mb-1 text-success">{{ $stats['services'] ?? 0 }}</h5>
                                <small class="text-muted">Layanan</small>
                            </div>
                        </div>
                        <div class="col-6">
                            <div class="stat-item text-center p-2 bg-info bg-opacity-10 rounded">
                                <h5 class="mb-1 text-info">{{ $stats['family_members'] ?? 0 }}</h5>
                                <small class="text-muted">Anggota KK</small>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Location Map -->
        @if($penduduk->latitude && $penduduk->longitude)
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-map me-2"></i>
                    Lokasi Rumah
                </h6>
            </div>
            <div class="card-body">
                <div id="locationMap" style="height: 250px; border-radius: 8px;"></div>
                <div class="mt-3 text-center">
                    <button type="button" class="btn btn-outline-primary btn-sm" onclick="getDirections()">
                        <i class="fas fa-route me-2"></i>
                        Petunjuk Arah
                    </button>
                </div>
            </div>
        </div>
        @endif
        
        <!-- Quick Actions -->
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-bolt me-2"></i>
                    Aksi Cepat
                </h6>
            </div>
            <div class="card-body">
                <div class="d-grid gap-2">
                    <button type="button" class="btn btn-outline-success" onclick="createSurat()">
                        <i class="fas fa-file-plus me-2"></i>
                        Buat Surat
                    </button>
                    <button type="button" class="btn btn-outline-info" onclick="addBantuan()">
                        <i class="fas fa-hand-holding-heart me-2"></i>
                        Tambah Bantuan
                    </button>
                    <button type="button" class="btn btn-outline-warning" onclick="updateData()">
                        <i class="fas fa-sync me-2"></i>
                        Update Data
                    </button>
                    <button type="button" class="btn btn-outline-danger" onclick="reportProblem()">
                        <i class="fas fa-flag me-2"></i>
                        Laporkan Masalah
                    </button>
                </div>
            </div>
        </div>
        
        <!-- Recent Activity -->
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-history me-2"></i>
                    Aktivitas Terbaru
                </h6>
            </div>
            <div class="card-body">
                <div class="activity-timeline">
                    @foreach($recent_activities ?? [] as $activity)
                    <div class="activity-item d-flex mb-3">
                        <div class="activity-icon bg-{{ $activity['color'] }} rounded-circle me-2" style="width: 32px; height: 32px;">
                            <i class="fas fa-{{ $activity['icon'] }}"></i>
                        </div>
                        <div class="activity-content">
                            <p class="mb-1 small">{{ $activity['message'] }}</p>
                            <small class="text-muted">{{ $activity['time'] }}</small>
                        </div>
                    </div>
                    @endforeach
                </div>
            </div>
        </div>
        
        <!-- Password Management Section -->
        <div class="card mt-4">
            <div class="card-header">
                <h6 class="card-title mb-0">
                    <i class="fas fa-lock me-2"></i>
                    Manajemen Password Portal Warga
                </h6>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <div class="mb-3">
                            <label class="form-label">Status Password</label>
                            <div>
                                @if($penduduk->hasPassword())
                                    @if($penduduk->mustChangePassword())
                                        <span class="badge bg-warning">
                                            <i class="fas fa-exclamation-triangle me-1"></i>
                                            Harus Diganti
                                        </span>
                                    @else
                                        <span class="badge bg-success">
                                            <i class="fas fa-check-circle me-1"></i>
                                            Aktif
                                        </span>
                                    @endif
                                @else
                                    <span class="badge bg-danger">
                                        <i class="fas fa-times-circle me-1"></i>
                                        Belum Diatur
                                    </span>
                                @endif
                            </div>
                        </div>
                        
                        @if($penduduk->password_set_at)
                        <div class="mb-3">
                            <label class="form-label">Password Diatur Pada</label>
                            <div class="text-muted">
                                {{ $penduduk->password_set_at->format('d F Y, H:i') }} WIB
                            </div>
                        </div>
                        @endif
                        
                        @if($penduduk->last_login_at)
                        <div class="mb-3">
                            <label class="form-label">Login Terakhir</label>
                            <div class="text-muted">
                                {{ $penduduk->last_login_at->format('d F Y, H:i') }} WIB
                                <small class="text-muted">({{ $penduduk->last_login_at->diffForHumans() }})</small>
                            </div>
                        </div>
                        @endif
                    </div>
                    
                    <div class="col-md-6">
                        <div class="d-grid gap-2">
                            @if(!$penduduk->hasPassword())
                                <button type="button" class="btn btn-success" onclick="generateDefaultPassword('{{ $penduduk->id }}')">
                                    <i class="fas fa-key me-2"></i>
                                    Generate Password Default
                                </button>
                            @else
                                <button type="button" class="btn btn-warning" onclick="resetPassword('{{ $penduduk->id }}')">
                                    <i class="fas fa-redo me-2"></i>
                                    Reset Password
                                </button>
                            @endif
                            
                            <button type="button" class="btn btn-info" onclick="showPasswordModal('{{ $penduduk->id }}', '{{ $penduduk->nama }}', '{{ $penduduk->nik }}')">
                                <i class="fas fa-edit me-2"></i>
                                Atur Password Custom
                            </button>
                            
                            @if($penduduk->hasPassword())
                                <button type="button" class="btn btn-outline-secondary" onclick="forcePasswordChange('{{ $penduduk->id }}')">
                                    <i class="fas fa-exclamation-triangle me-2"></i>
                                    Paksa Ganti Password
                                </button>
                            @endif
                        </div>
                    </div>
                </div>
                
                <div class="alert alert-info mt-3">
                    <i class="fas fa-info-circle me-2"></i>
                    <strong>Informasi:</strong>
                    <ul class="mb-0 mt-2">
                        <li>Password default adalah 6 digit terakhir NIK ({{ substr($penduduk->nik, -6) }})</li>
                        <li>Warga dapat login ke Portal Layanan Mandiri menggunakan NIK dan password</li>
                        <li>Password dapat diubah oleh admin atau warga sendiri setelah login</li>
                    </ul>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Photo Modal -->
<div class="modal fade" id="photoModal" tabindex="-1">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Foto {{ $penduduk->nama }}</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body text-center">
                <img id="modalPhoto" src="" class="img-fluid" alt="Foto Penduduk">
            </div>
        </div>
    </div>
</div>

<!-- Create Service Modal -->
<div class="modal fade" id="createServiceModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Buat Layanan Baru</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="createServiceForm">
                    <div class="mb-3">
                        <label class="form-label">Jenis Layanan</label>
                        <select class="form-select" name="service_type" required>
                            <option value="">Pilih Layanan</option>
                            <option value="surat_domisili">Surat Domisili</option>
                            <option value="surat_skck">SKCK</option>
                            <option value="surat_usaha">Surat Usaha</option>
                            <option value="bantuan_sosial">Bantuan Sosial</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label class="form-label">Keterangan</label>
                        <textarea class="form-control" name="keterangan" rows="3" 
                                  placeholder="Keterangan tambahan..."></textarea>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-primary" onclick="submitService()">Buat Layanan</button>
            </div>
        </div>
    </div>
</div>

<!-- Password Management Modal -->
<div class="modal fade" id="passwordModal" tabindex="-1">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title">Atur Password Custom</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body">
                <form id="passwordForm">
                    <input type="hidden" id="penduduk_id" name="penduduk_id">
                    
                    <div class="mb-3">
                        <label class="form-label">Nama Penduduk</label>
                        <input type="text" id="penduduk_nama" class="form-control" readonly>
                    </div>
                    
                    <div class="mb-3">
                        <label class="form-label">NIK</label>
                        <input type="text" id="penduduk_nik" class="form-control" readonly>
                    </div>
                    
                    <div class="mb-3">
                        <label for="new_password" class="form-label">Password Baru</label>
                        <div class="input-group">
                            <input type="password" class="form-control" id="new_password" name="new_password" required minlength="6">
                            <button type="button" class="btn btn-outline-secondary" onclick="togglePasswordVisibility()">
                                <i class="fas fa-eye" id="toggleIcon"></i>
                            </button>
                        </div>
                        <small class="text-muted">Minimal 6 karakter</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="confirm_password" class="form-label">Konfirmasi Password</label>
                        <input type="password" class="form-control" id="confirm_password" name="confirm_password" required>
                    </div>
                    
                    <div class="form-check mb-3">
                        <input class="form-check-input" type="checkbox" id="must_change" name="must_change">
                        <label class="form-check-label" for="must_change">
                            Paksa warga mengganti password saat login pertama
                        </label>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Batal</button>
                <button type="button" class="btn btn-primary" onclick="setCustomPassword()">
                    <i class="fas fa-save me-2"></i>Simpan Password
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
    /* Page Header Styles */
    .page-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 2rem;
        border-radius: 12px;
        margin-bottom: 2rem;
    }
    
    .page-icon {
        width: 60px;
        height: 60px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 24px;
    }
    
    .page-title {
        font-size: 2rem;
        font-weight: 600;
        margin: 0;
    }
    
    .page-subtitle {
        font-size: 1rem;
        opacity: 0.9;
    }
    
    /* Profile Card Styles */
    .profile-card {
        border: none;
        border-radius: 16px;
    }
    
    .profile-photo-container {
        position: relative;
        display: inline-block;
    }
    
    .profile-photo {
        width: 180px;
        height: 180px;
        object-fit: cover;
        cursor: pointer;
        border-radius: 50%;
        border: 4px solid #fff;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        transition: transform 0.3s ease;
    }
    
    .profile-photo:hover {
        transform: scale(1.05);
    }
    
    .profile-photo-placeholder {
        width: 180px;
        height: 180px;
        border: 3px dashed #dee2e6;
        border-radius: 50%;
        display: flex;
        align-items: center;
        justify-content: center;
        margin: 0 auto;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    }
    
    .avatar-initials {
        font-size: 3rem;
        font-weight: 600;
        color: #6c757d;
    }
    
    .profile-status-badge {
        position: absolute;
        bottom: 10px;
        right: 10px;
    }
    
    .profile-name {
        font-size: 1.75rem;
        font-weight: 600;
        color: #2c3e50;
    }
    
    .profile-nik {
        font-size: 1rem;
        font-family: 'Courier New', monospace;
    }
    
    .profile-badges .badge {
        font-size: 0.875rem;
        padding: 0.5rem 0.75rem;
    }
    
    /* Section Styles */
    .section-header {
        border-bottom: 2px solid #f8f9fa;
        padding-bottom: 1rem;
    }
    
    .section-title {
        font-size: 1.25rem;
        font-weight: 600;
        color: #2c3e50;
        margin: 0;
    }
    
    /* Info Group Styles */
    .info-group {
        background: #f8f9fa;
        border-radius: 12px;
        padding: 1.5rem;
        height: 100%;
    }
    
    .info-group-title {
        font-size: 1rem;
        font-weight: 600;
        color: #495057;
        margin-bottom: 1rem;
        padding-bottom: 0.5rem;
        border-bottom: 2px solid #dee2e6;
    }
    
    .info-list {
        display: flex;
        flex-direction: column;
        gap: 1rem;
    }
    
    .info-item {
        display: flex;
        flex-direction: column;
        gap: 0.25rem;
    }
    
    .info-label {
        font-size: 0.875rem;
        font-weight: 500;
        color: #6c757d;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .info-value {
        font-size: 1rem;
        font-weight: 500;
        color: #2c3e50;
        word-break: break-word;
    }
    
    /* Address Styles */
    .address-details {
        background: #f8f9fa;
        border-radius: 12px;
        padding: 1.5rem;
        height: 100%;
    }
    
    .coordinates-card {
        border: 1px solid #dee2e6;
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
    }
    
    .coordinate-item {
        padding: 0.5rem;
    }
    
    .coordinate-value {
        font-size: 1rem;
        font-weight: 600;
        color: #2c3e50;
        font-family: 'Courier New', monospace;
    }
    
    .location-placeholder {
        border: 2px dashed #dee2e6;
        border-radius: 12px;
        background: #f8f9fa;
    }
    
    .placeholder-icon {
        opacity: 0.5;
    }
    
    /* Family Card Styles */
    .family-header {
        border: 1px solid #e9ecef;
    }
    
    .member-card {
        transition: all 0.3s ease;
        border: 1px solid #dee2e6;
        background: white;
    }
    
    .member-card:hover {
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        transform: translateY(-2px);
    }
    
    .member-avatar {
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 16px;
    }
    
    .member-name {
        font-size: 1rem;
        color: #2c3e50;
    }
    
    .member-relation {
        font-weight: 500;
        color: #6c757d !important;
    }
    
    .member-details {
        color: #868e96 !important;
    }
    
    .empty-state {
        padding: 2rem;
    }
    
    .avatar {
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 12px;
    }
    
    /* Activity Styles */
    .activity-icon, .service-icon {
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 14px;
        min-width: 40px;
        min-height: 40px;
    }
    
    .services-timeline {
        max-height: 400px;
        overflow-y: auto;
    }
    
    /* Statistics Styles */
    .stat-item {
        transition: all 0.3s ease;
        border-radius: 12px;
        border: 1px solid transparent;
    }
    
    .stat-item:hover {
        transform: translateY(-3px);
        box-shadow: 0 4px 12px rgba(0,0,0,0.1);
        border-color: #dee2e6;
    }
    
    /* Utility Classes */
    .bg-pink {
        background-color: #e83e8c !important;
    }
    
    .shadow-sm {
        box-shadow: 0 2px 8px rgba(0,0,0,0.08) !important;
    }
    
    /* Responsive Design */
    @media (max-width: 768px) {
        .page-header {
            padding: 1.5rem;
        }
        
        .page-title {
            font-size: 1.5rem;
        }
        
        .profile-photo, .profile-photo-placeholder {
            width: 150px;
            height: 150px;
        }
        
        .avatar-initials {
            font-size: 2.5rem;
        }
        
        .profile-name {
            font-size: 1.5rem;
        }
        
        .info-group {
            padding: 1rem;
        }
        
        .btn-toolbar {
            flex-direction: column;
            gap: 0.5rem;
        }
        
        .btn-group {
            width: 100%;
        }
    }
    
    /* Animation */
    @keyframes fadeInUp {
        from {
            opacity: 0;
            transform: translateY(20px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
    
    .card {
        animation: fadeInUp 0.6s ease-out;
    }
    
    .card:nth-child(2) { animation-delay: 0.1s; }
    .card:nth-child(3) { animation-delay: 0.2s; }
    .card:nth-child(4) { animation-delay: 0.3s; }
</style>
@endpush

@push('scripts')
<script>
let map;
const pendudukId = {{ $penduduk->id }};
const latitude = {{ $penduduk->latitude ?? 'null' }};
const longitude = {{ $penduduk->longitude ?? 'null' }};

$(document).ready(function() {
    // Initialize map if coordinates exist
    if (latitude && longitude) {
        initMap();
    }
});

// Initialize map
function initMap() {
    map = L.map('locationMap').setView([latitude, longitude], 16);
    
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '© OpenStreetMap contributors'
    }).addTo(map);
    
    // Add marker
    const marker = L.marker([latitude, longitude]).addTo(map);
    marker.bindPopup(`
        <strong>{{ $penduduk->nama }}</strong><br>
        {{ $penduduk->alamat }}<br>
        Dusun {{ $penduduk->dusun }}, RT {{ $penduduk->rt }}/RW {{ $penduduk->rw }}
    `).openPopup();
}

// Show photo modal
function showPhotoModal(src) {
    document.getElementById('modalPhoto').src = src;
    $('#photoModal').modal('show');
}

// Send WhatsApp
function sendWhatsApp() {
    @if($penduduk->no_wa)
    const message = encodeURIComponent('Halo {{ $penduduk->nama }}, ');
    window.open(`https://wa.me/{{ $penduduk->no_wa }}?text=${message}`, '_blank');
    @else
    showToast('Nomor WhatsApp tidak tersedia', 'warning');
    @endif
}

// Show on map
function showOnMap() {
    @if($penduduk->latitude && $penduduk->longitude)
    window.location.href = `/admin/peta?lat={{ $penduduk->latitude }}&lng={{ $penduduk->longitude }}&zoom=16&highlight=penduduk_{{ $penduduk->id }}`;
    @else
    showToast('Koordinat rumah belum diset', 'warning');
    @endif
}

// Open in Google Maps
function openInGoogleMaps() {
    @if($penduduk->latitude && $penduduk->longitude)
    window.open(`https://www.google.com/maps?q={{ $penduduk->latitude }},{{ $penduduk->longitude }}`, '_blank');
    @else
    showToast('Koordinat rumah belum diset', 'warning');
    @endif
}

// Get directions
function getDirections() {
    @if($penduduk->latitude && $penduduk->longitude)
    window.open(`https://www.google.com/maps/dir/?api=1&destination={{ $penduduk->latitude }},{{ $penduduk->longitude }}`, '_blank');
    @else
    showToast('Koordinat rumah belum diset', 'warning');
    @endif
}

// Print functions
function printKTP() {
    window.open(`/admin/penduduk/{{ $penduduk->id }}/print-ktp`, '_blank');
}

function printKK() {
    @if($penduduk->keluarga)
    window.open(`/admin/keluarga/{{ $penduduk->keluarga->id }}/print-kk`, '_blank');
    @else
    showToast('Data keluarga tidak tersedia', 'warning');
    @endif
}

function printProfile() {
    window.open(`/admin/penduduk/{{ $penduduk->id }}/print-profile`, '_blank');
}

// Quick actions
function createSurat() {
    $('#createServiceModal').modal('show');
}

function addBantuan() {
    window.location.href = `/admin/bantuan/create?penduduk_id={{ $penduduk->id }}`;
}

function updateData() {
    window.location.href = `/admin/penduduk/{{ $penduduk->id }}/edit`;
}

function reportProblem() {
    window.location.href = `/admin/pengaduan/create?penduduk_id={{ $penduduk->id }}`;
}

// Submit service
function submitService() {
    const form = document.getElementById('createServiceForm');
    const formData = new FormData(form);
    formData.append('penduduk_id', pendudukId);
    
    showLoading();
    
    fetch('/admin/services', {
        method: 'POST',
        headers: {
            'X-CSRF-TOKEN': $('meta[name="csrf-token"]').attr('content')
        },
        body: formData
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            showToast('Layanan berhasil dibuat', 'success');
            $('#createServiceModal').modal('hide');
            setTimeout(() => location.reload(), 1500);
        } else {
            showToast(data.message || 'Gagal membuat layanan', 'error');
        }
    })
    .catch(error => {
        console.error('Error:', error);
        showToast('Terjadi kesalahan', 'error');
    })
    .finally(() => {
        hideLoading();
    });
}

// Password Management Functions
function generateDefaultPassword(pendudukId) {
    if (confirm('Generate password default untuk penduduk ini?\n\nPassword akan diatur menjadi 6 digit terakhir NIK.')) {
        fetch(`/admin/penduduk/${pendudukId}/generate-password`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert(`Password berhasil digenerate!\n\nPassword: ${data.password}\n\nSilakan berikan password ini kepada warga.`);
                location.reload();
            } else {
                alert('Gagal generate password: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Terjadi kesalahan saat generate password');
        });
    }
}

function resetPassword(pendudukId) {
    if (confirm('Reset password ke default (6 digit terakhir NIK)?\n\nPassword lama akan dihapus dan diganti dengan password default.')) {
        fetch(`/admin/penduduk/${pendudukId}/reset-password`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert(`Password berhasil direset!\n\nPassword baru: ${data.password}\n\nSilakan berikan password ini kepada warga.`);
                location.reload();
            } else {
                alert('Gagal reset password: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Terjadi kesalahan saat reset password');
        });
    }
}

function showPasswordModal(pendudukId, nama, nik) {
    document.getElementById('penduduk_id').value = pendudukId;
    document.getElementById('penduduk_nama').value = nama;
    document.getElementById('penduduk_nik').value = nik;
    document.getElementById('new_password').value = '';
    document.getElementById('confirm_password').value = '';
    document.getElementById('must_change').checked = false;
    
    new bootstrap.Modal(document.getElementById('passwordModal')).show();
}

function setCustomPassword() {
    const form = document.getElementById('passwordForm');
    const formData = new FormData(form);
    
    const newPassword = document.getElementById('new_password').value;
    const confirmPassword = document.getElementById('confirm_password').value;
    
    if (newPassword !== confirmPassword) {
        alert('Password dan konfirmasi password tidak sama!');
        return;
    }
    
    if (newPassword.length < 6) {
        alert('Password minimal 6 karakter!');
        return;
    }
    
    const pendudukId = document.getElementById('penduduk_id').value;
    
    fetch(`/admin/penduduk/${pendudukId}/set-password`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
        },
        body: JSON.stringify({
            password: newPassword,
            must_change: document.getElementById('must_change').checked
        })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            alert('Password berhasil diatur!');
            bootstrap.Modal.getInstance(document.getElementById('passwordModal')).hide();
            location.reload();
        } else {
            alert('Gagal mengatur password: ' + data.message);
        }
    })
    .catch(error => {
        console.error('Error:', error);
        alert('Terjadi kesalahan saat mengatur password');
    });
}

function forcePasswordChange(pendudukId) {
    if (confirm('Paksa warga untuk mengganti password saat login berikutnya?')) {
        fetch(`/admin/penduduk/${pendudukId}/force-password-change`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('Berhasil! Warga akan diminta mengganti password saat login berikutnya.');
                location.reload();
            } else {
                alert('Gagal: ' + data.message);
            }
        })
        .catch(error => {
            console.error('Error:', error);
            alert('Terjadi kesalahan');
        });
    }
}

function togglePasswordVisibility() {
    const passwordInput = document.getElementById('new_password');
    const toggleIcon = document.getElementById('toggleIcon');
    
    if (passwordInput.type === 'password') {
        passwordInput.type = 'text';
        toggleIcon.className = 'fas fa-eye-slash';
    } else {
        passwordInput.type = 'password';
        toggleIcon.className = 'fas fa-eye';
    }
}
</script>
@endpush 
