<!DOCTYPE html>
<html lang="id">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Peta Desa - Sistem Informasi Desa</title>
    
    <!-- PWA Meta Tags -->
    <meta name="theme-color" content="#10B981">
    <meta name="mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-capable" content="yes">
    <meta name="apple-mobile-web-app-status-bar-style" content="default">
    <meta name="apple-mobile-web-app-title" content="Sistem Desa">
    <link rel="manifest" href="/manifest.json">
    
    <!-- Favicon -->
    <link rel="icon" type="image/x-icon" href="/favicon.ico">
    
    <!-- CSS -->
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    fontFamily: {
                        'inter': ['Inter', 'system-ui', 'sans-serif'],
                    },
                    colors: {
                        primary: {
                            50: '#ecfdf5',
                            500: '#10b981',
                            600: '#059669',
                            700: '#047857'
                        }
                    }
                }
            }
        }
    </script>
    
    <link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css">
    <script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css">
    <!-- FontAwesome Fallback -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" integrity="sha512-iecdLmaskl7CVkqkXNQ/ZH/XLlvWZOJyj7Yy7tcenmpD1ypASozpmT/E0iPtmFIB46ZmdtAc9eNBvH0H/ZpiBw==" crossorigin="anonymous" referrerpolicy="no-referrer" />
    <link rel="stylesheet" href="https://use.fontawesome.com/releases/v6.4.0/css/all.css" crossorigin="anonymous" />
    
    <style>
        @import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
        
        body {
            font-family: 'Inter', system-ui, sans-serif;
        }
        
        .mobile-card {
            background: rgba(255, 255, 255, 0.95);
            backdrop-filter: blur(10px);
            border-radius: 1rem;
            padding: 1.5rem;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
            border: 1px solid #f3f4f6;
        }
        
        .gradient-bg {
            background: linear-gradient(135deg, #10b981 0%, #059669 100%);
        }
        
        .map-container {
            height: 50vh;
            min-height: 350px;
            max-height: 450px;
        }
        
        .legend-item {
            display: flex;
            align-items: center;
            gap: 0.5rem;
            font-size: 0.875rem;
        }
        
        .marker-dot {
            width: 0.75rem;
            height: 0.75rem;
            border-radius: 50%;
        }
        
        .info-panel {
            position: absolute;
            top: 1rem;
            left: 1rem;
            background: rgba(255, 255, 255, 0.9);
            backdrop-filter: blur(10px);
            border-radius: 0.75rem;
            padding: 1rem;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            z-index: 1000;
            max-width: 20rem;
        }
        
        .floating-buttons {
            position: absolute;
            bottom: 1.5rem;
            right: 1rem;
            gap: 0.5rem;
            display: flex;
            flex-direction: column;
            z-index: 1000;
        }
        
        .floating-btn {
            width: 3rem;
            height: 3rem;
            background: white;
            border-radius: 50%;
            box-shadow: 0 10px 15px -3px rgba(0, 0, 0, 0.1);
            display: flex;
            align-items: center;
            justify-content: center;
            color: #6b7280;
            transition: all 0.3s;
        }
        
        .floating-btn:hover {
            background: #eff6ff;
            color: #2563eb;
        }
        
        /* Responsive adjustments */
        @media (max-width: 640px) {
            .map-container {
                height: 45vh;
                min-height: 300px;
                max-height: 400px;
            }
            
            .floating-buttons {
                bottom: 1rem;
                right: 0.5rem;
            }
            
            .floating-btn {
                width: 2.5rem;
                height: 2.5rem;
            }
        }
        
        @media (max-height: 600px) {
            .map-container {
                height: 40vh;
                min-height: 250px;
            }
        }
        
        /* FontAwesome fallback with !important for navigation */
        .fas.fa-home:before, .fa.fa-home:before { content: "🏠" !important; font-family: system-ui !important; }
        .fas.fa-newspaper:before, .fa.fa-newspaper:before { content: "📰" !important; font-family: system-ui !important; }
        .fas.fa-map-marked-alt:before, .fa.fa-map-marked-alt:before { content: "🗺️" !important; font-family: system-ui !important; }
        .fas.fa-store:before, .fa.fa-store:before { content: "🏪" !important; font-family: system-ui !important; }
        .fas.fa-map:before, .fa.fa-map:before { content: "🗺️" !important; font-family: system-ui !important; }
        .fas.fa-search:before, .fa.fa-search:before { content: "🔍" !important; font-family: system-ui !important; }
        .fas.fa-location-arrow:before, .fa.fa-location-arrow:before { content: "🧭" !important; font-family: system-ui !important; }
        .fas.fa-expand:before, .fa.fa-expand:before { content: "🔍" !important; font-family: system-ui !important; }
        .fas.fa-compress:before, .fa.fa-compress:before { content: "🔍" !important; font-family: system-ui !important; }
        .fas.fa-info:before, .fa.fa-info:before { content: "ℹ️" !important; font-family: system-ui !important; }
        
        /* Force emoji display for all FontAwesome icons */
        .fas:before, .fab:before, .fa:before {
            font-weight: normal !important;
            font-style: normal !important;
            text-rendering: auto !important;
            -webkit-font-smoothing: antialiased !important;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    <!-- Mobile App Container -->
    <div class="max-w-md mx-auto bg-white min-h-screen shadow-xl">
        
        <!-- Header/Status Bar -->
        <div class="gradient-bg text-white px-6 py-8 relative overflow-hidden">
            <!-- Background Pattern -->
            <div class="absolute inset-0 opacity-10">
                <div class="absolute top-0 left-0 w-40 h-40 bg-white rounded-full -translate-x-20 -translate-y-20"></div>
                <div class="absolute bottom-0 right-0 w-32 h-32 bg-white rounded-full translate-x-16 translate-y-16"></div>
            </div>
            
            <!-- Header Content -->
            <div class="relative z-10">
                <div class="flex items-center justify-between mb-6">
                    <div class="flex items-center space-x-3">
                        <button onclick="history.back()" class="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center backdrop-blur-md">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 19l-7-7 7-7"/>
                            </svg>
                        </button>
                        <div>
                            <h1 class="text-xl font-bold">Peta Desa</h1>
                            <p class="text-white/80 text-sm">Peta interaktif wilayah</p>
                        </div>
                    </div>
                    
                    <div class="flex items-center space-x-2">
                        <button onclick="toggleLegend()" class="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center backdrop-blur-md">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                            </svg>
                        </button>
                        <button onclick="toggleSearch()" class="w-10 h-10 bg-white/20 rounded-full flex items-center justify-center backdrop-blur-md">
                            <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                            </svg>
                        </button>
                    </div>
                </div>
                
                <!-- Service Info -->
                <div class="text-center">
                    <h2 class="text-lg font-semibold mb-2">Peta Interaktif Desa</h2>
                    <p class="text-white/90 text-sm">Jelajahi lokasi dan fasilitas desa</p>
                </div>
            </div>
        </div>

        <!-- Search Bar (Hidden by default) -->
        <div id="searchBar" class="hidden px-6 py-4 bg-white border-b border-gray-100">
            <div class="relative">
                <input type="text" id="searchInput" 
                       placeholder="Cari lokasi, RT/RW, nama jalan..." 
                       class="w-full px-4 py-3 pr-12 bg-gray-50 border border-gray-200 rounded-xl focus:outline-none focus:ring-2 focus:ring-green-500 focus:border-transparent">
                <button onclick="searchLocation()" class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-400 hover:text-green-600">
                    <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M21 21l-6-6m2-5a7 7 0 11-14 0 7 7 0 0114 0z"/>
                    </svg>
                </button>
            </div>
        </div>

        <!-- Map Container -->
        <div class="relative">
            <div id="mapContainer" class="map-container">
                <!-- Map will be initialized here -->
            </div>
            
            <!-- Map Info Panel -->
            <div id="infoPanel" class="info-panel hidden">
                <h3 class="font-semibold text-gray-800 mb-2">Informasi Lokasi</h3>
                <div id="locationInfo" class="space-y-1">
                    <p class="text-sm text-gray-600">Klik pada marker untuk melihat detail lokasi</p>
                </div>
            </div>
            
            <!-- Legend Panel -->
            <div id="legendPanel" class="info-panel top-auto bottom-4 hidden">
                <h3 class="font-semibold text-gray-800 mb-3">Legenda</h3>
                <div class="space-y-2">
                    <div class="legend-item">
                        <div class="marker-dot" style="background-color: #DC2626;"></div>
                        <span>Kantor Desa</span>
                    </div>
                    <div class="legend-item">
                        <div class="marker-dot" style="background-color: #2563EB;"></div>
                        <span>Fasilitas Umum</span>
                    </div>
                    <div class="legend-item">
                        <div class="marker-dot" style="background-color: #7C3AED;"></div>
                        <span>UMKM</span>
                    </div>
                    <div class="legend-item">
                        <div class="marker-dot" style="background-color: #EA580C;"></div>
                        <span>Wisata</span>
                    </div>
                </div>
            </div>
            
            <!-- Floating Action Buttons -->
            <div class="floating-buttons">
                <button onclick="centerMap()" class="floating-btn" title="Kembali ke Pusat">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M17.657 16.657L13.414 20.9a1.998 1.998 0 01-2.827 0l-4.244-4.243a8 8 0 1111.314 0z"/>
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M15 11a3 3 0 11-6 0 3 3 0 016 0z"/>
                    </svg>
                </button>
                <button onclick="getCurrentLocation()" class="floating-btn" title="Lokasi Saya">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8"/>
                    </svg>
                </button>
                <button onclick="toggleFullscreen()" class="floating-btn" title="Layar Penuh">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M4 8V4m0 0h4M4 4l5 5m11-1V4m0 0h-4m4 0l-5 5M4 16v4m0 0h4m-4 0l5-5m11 5l-5-5m5 5v-4m0 4h-4"/>
                    </svg>
                </button>
            </div>
        </div>

        <!-- Statistics & Quick Info -->
        <div class="px-6 py-4">
            <div class="grid grid-cols-3 gap-4 mb-4">

                
                <div class="mobile-card text-center">
                    <div class="w-10 h-10 bg-green-100 rounded-full flex items-center justify-center mx-auto mb-2">
                        <svg class="w-5 h-5 text-green-600" fill="currentColor" viewBox="0 0 20 20">
                            <path d="M10.394 2.08a1 1 0 00-.788 0l-7 3a1 1 0 000 1.84L5.25 8.051a.999.999 0 01.356-.257l4-1.714a1 1 0 11.788 1.838L7.667 9.088l1.94.831a1 1 0 00.787 0l7-3a1 1 0 000-1.838l-7-3zM3.31 9.397L5 10.12v4.102a8.969 8.969 0 00-1.05-.174 1 1 0 01-.89-.89 11.115 11.115 0 01.25-3.762zM9.3 16.573A9.026 9.026 0 007 14.935v-3.957l1.818.78a3 3 0 002.364 0l5.508-2.361a11.026 11.026 0 01.25 3.762 1 1 0 01-.89.89 8.968 8.968 0 00-5.35 2.524 1 1 0 01-1.4 0zM6 18a1 1 0 001-1v-2.065a8.935 8.935 0 00-2-.712V17a1 1 0 001 1z"/>
                        </svg>
                    </div>
                    <div class="text-lg font-bold text-gray-800">0</div>
                    <div class="text-xs text-gray-500">Fasilitas Umum</div>
                </div>
                
                <div class="mobile-card text-center">
                    <div class="w-10 h-10 bg-purple-100 rounded-full flex items-center justify-center mx-auto mb-2">
                        <svg class="w-5 h-5 text-purple-600" fill="currentColor" viewBox="0 0 20 20">
                            <path d="M3 1a1 1 0 000 2h1.22l.305 1.222a.997.997 0 00.01.042l1.358 5.43-.893.892C3.74 11.846 4.632 14 6.414 14H15a1 1 0 000-2H6.414l1-1H14a1 1 0 00.894-.553l3-6A1 1 0 0017 3H6.28l-.31-1.243A1 1 0 005 1H3zM16 16.5a1.5 1.5 0 11-3 0 1.5 1.5 0 013 0zM6.5 18a1.5 1.5 0 100-3 1.5 1.5 0 000 3z"/>
                        </svg>
                    </div>
                    <div class="text-lg font-bold text-gray-800">0</div>
                    <div class="text-xs text-gray-500">UMKM</div>
                </div>
                
                <div class="mobile-card text-center">
                    <div class="w-10 h-10 bg-orange-100 rounded-full flex items-center justify-center mx-auto mb-2">
                        <svg class="w-5 h-5 text-orange-600" fill="currentColor" viewBox="0 0 20 20">
                            <path fill-rule="evenodd" d="M4 3a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V5a2 2 0 00-2-2H4zm12 12H4l4-8 3 6 2-4 3 6z"/>
                        </svg>
                    </div>
                    <div class="text-lg font-bold text-gray-800">0</div>
                    <div class="text-xs text-gray-500">Wisata</div>
                </div>
            </div>
        </div>

        <!-- Filter Categories -->
        <div class="px-6 pb-4">
            <h3 class="text-lg font-semibold text-gray-800 mb-3">Filter Lokasi</h3>
            <div class="flex space-x-2 overflow-x-auto pb-2">
                <button onclick="filterMarkers('all')" class="filter-btn active flex-shrink-0 px-4 py-2 bg-blue-600 text-white text-sm font-medium rounded-full transition-all">
                    Semua
                </button>
                <button onclick="filterMarkers('kantor')" class="filter-btn flex-shrink-0 px-4 py-2 bg-white text-gray-600 text-sm font-medium rounded-full border border-gray-200 hover:border-red-300 transition-all">
                    Kantor
                </button>
                <button onclick="filterMarkers('fasilitas')" class="filter-btn flex-shrink-0 px-4 py-2 bg-white text-gray-600 text-sm font-medium rounded-full border border-gray-200 hover:border-blue-300 transition-all">
                    Fasilitas
                </button>
                <button onclick="filterMarkers('umkm')" class="filter-btn flex-shrink-0 px-4 py-2 bg-white text-gray-600 text-sm font-medium rounded-full border border-gray-200 hover:border-purple-300 transition-all">
                    UMKM
                </button>
                <button onclick="filterMarkers('wisata')" class="filter-btn flex-shrink-0 px-4 py-2 bg-white text-gray-600 text-sm font-medium rounded-full border border-gray-200 hover:border-orange-300 transition-all">
                    Wisata
                </button>
            </div>
        </div>

        <!-- Bottom Navigation -->
        <div class="fixed bottom-0 left-1/2 transform -translate-x-1/2 w-full max-w-md bg-white border-t border-gray-100 px-4 py-3 z-50 shadow-lg backdrop-blur-sm bg-white/95">
            <div class="flex justify-around items-center">
                <button onclick="navigateToService('/')" class="flex flex-col items-center space-y-1.5 text-gray-500 hover:text-blue-600 transition-all duration-200">
                    <div class="p-2 rounded-xl hover:bg-blue-50">
                        <i class="fas fa-home text-lg"></i>
                    </div>
                    <span class="text-xs">Beranda</span>
                </button>
                
                <button onclick="navigateToService('berita')" class="flex flex-col items-center space-y-1.5 text-gray-500 hover:text-blue-600 transition-all duration-200">
                    <div class="p-2 rounded-xl hover:bg-blue-50">
                        <i class="fas fa-newspaper text-lg"></i>
                    </div>
                    <span class="text-xs">Berita</span>
                </button>
                
                <button onclick="navigateToService('wisata')" class="flex flex-col items-center space-y-1.5 text-gray-500 hover:text-blue-600 transition-all duration-200">
                    <div class="p-2 rounded-xl hover:bg-blue-50">
                        <i class="fas fa-map-marked-alt text-lg"></i>
                    </div>
                    <span class="text-xs">Wisata</span>
                </button>
                
                <button onclick="navigateToService('umkm')" class="flex flex-col items-center space-y-1.5 text-gray-500 hover:text-blue-600 transition-all duration-200">
                    <div class="p-2 rounded-xl hover:bg-blue-50">
                        <i class="fas fa-store text-lg"></i>
                    </div>
                    <span class="text-xs">UMKM</span>
                </button>
                
                <button class="flex flex-col items-center space-y-1.5 text-blue-600 transition-all duration-200">
                    <div class="bg-blue-50 p-2 rounded-xl">
                        <i class="fas fa-map text-lg"></i>
                    </div>
                    <span class="text-xs font-semibold">Peta</span>
                </button>
            </div>
        </div>
        
        <!-- Spacer for bottom navigation -->
        <div class="h-24"></div>
    </div>

    <!-- JavaScript -->
    <script>
        let map;
        let markers = [];
        let markerGroups = {
            kantor: [],
            fasilitas: [],
            umkm: [],
            wisata: [],
            all: []
        };

        // Initialize map
        function initMap() {
            // Default center (example coordinates - adjust to your village location)
            const centerLat = -7.257472;
            const centerLng = 112.637416;
            
            map = L.map('mapContainer').setView([centerLat, centerLng], 15);
            
            // Add tile layer
            L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
                attribution: '© OpenStreetMap contributors'
            }).addTo(map);
            
            // Load markers from backend
            loadMarkersFromBackend();
        }

        // Load markers from backend API
        async function loadMarkersFromBackend() {
            try {
                // Fetch data from multiple API endpoints
                const [umkmResponse, wisataResponse, fasilitasResponse] = await Promise.all([
                    fetch('/api/v1/umkm-locations'),
                    fetch('/api/v1/wisata-locations'), 
                    fetch('/api/v1/fasilitas-locations')
                ]);

                const umkmData = await umkmResponse.json();
                const wisataData = await wisataResponse.json();
                const fasilitasData = await fasilitasResponse.json();

                // Process UMKM data
                if (umkmData.success && umkmData.data) {
                    umkmData.data.forEach(umkm => {
                        if (umkm.latitude && umkm.longitude) {
                            const data = {
                                lat: parseFloat(umkm.latitude),
                                lng: parseFloat(umkm.longitude),
                                type: 'umkm',
                                title: umkm.nama_umkm,
                                description: umkm.kategori || 'UMKM Desa',
                                foto: umkm.foto ? `/storage/${umkm.foto}` : 'https://via.placeholder.com/240x100/7C3AED/FFFFFF?text=UMKM'
                            };
                            addMarkerToMap(data);
                        }
                    });
                }

                // Process Wisata data
                if (wisataData.success && wisataData.data) {
                    wisataData.data.forEach(wisata => {
                        if (wisata.latitude && wisata.longitude) {
                            const data = {
                                lat: parseFloat(wisata.latitude),
                                lng: parseFloat(wisata.longitude),
                                type: 'wisata',
                                title: wisata.nama,
                                description: wisata.deskripsi || 'Destinasi Wisata',
                                foto: wisata.foto ? `/storage/${wisata.foto}` : 'https://via.placeholder.com/240x100/EA580C/FFFFFF?text=Wisata'
                            };
                            addMarkerToMap(data);
                        }
                    });
                }

                // Process Fasilitas data
                if (fasilitasData.success && fasilitasData.data) {
                    fasilitasData.data.forEach(fasilitas => {
                        const data = {
                            lat: parseFloat(fasilitas.latitude),
                            lng: parseFloat(fasilitas.longitude),
                            type: fasilitas.kategori === 'Kantor Desa' ? 'kantor' : 'fasilitas',
                            title: fasilitas.nama,
                            description: fasilitas.kategori || 'Fasilitas Umum',
                            foto: fasilitas.foto ? `/storage/${fasilitas.foto}` : 
                                  (fasilitas.kategori === 'Kantor Desa' ? 
                                   'https://via.placeholder.com/240x100/DC2626/FFFFFF?text=Kantor+Desa' :
                                   'https://via.placeholder.com/240x100/2563EB/FFFFFF?text=Fasilitas')
                        };
                        addMarkerToMap(data);
                    });
                }

                // Center map to show all markers
                if (markers.length > 0) {
                    const group = new L.featureGroup(markers);
                    map.fitBounds(group.getBounds().pad(0.1));
                }

            } catch (error) {
                console.error('Error loading markers from backend:', error);
                // Fallback to static data if API fails
                loadFallbackMarkers();
            }
        }

        // Add individual marker to map
        function addMarkerToMap(data) {
            const markerIcon = createCustomMarker(data.type);
            
            const marker = L.marker([data.lat, data.lng], { icon: markerIcon })
                .addTo(map)
                .bindPopup(createPopupContent(data));

            markers.push(marker);
            markerGroups[data.type].push(marker);
            markerGroups.all.push(marker);
        }

        // Fallback static data if API fails
        function loadFallbackMarkers() {
            const fallbackData = [
                {
                    lat: -6.7175,
                    lng: 107.1426,
                    type: 'kantor',
                    title: 'Kantor Desa Sirnajaya',
                    description: 'Balai Desa - Pusat Administrasi',
                    foto: 'https://via.placeholder.com/240x100/DC2626/FFFFFF?text=Kantor+Desa'
                },
                {
                    lat: -6.7185,
                    lng: 107.1436,
                    type: 'fasilitas',
                    title: 'Puskesmas Sukamakmur',
                    description: 'Layanan Kesehatan Masyarakat',
                    foto: 'https://via.placeholder.com/240x100/2563EB/FFFFFF?text=Puskesmas'
                }
            ];

            fallbackData.forEach(data => {
                addMarkerToMap(data);
            });
        }

        // Create custom marker icons for different types
        function createCustomMarker(type) {
            let color, iconHtml;
            
            switch(type) {
                case 'kantor':
                    color = '#DC2626'; // Red
                    iconHtml = '<i class="fas fa-building" style="color: white; font-size: 12px;"></i>';
                    break;
                case 'fasilitas':
                    color = '#2563EB'; // Blue
                    // Menggunakan ikon yang sama dengan card fasilitas (building dengan path)
                    iconHtml = `<svg style="color: white; width: 12px; height: 12px;" fill="currentColor" viewBox="0 0 20 20">
                        <path d="M10.394 2.08a1 1 0 00-.788 0l-7 3a1 1 0 000 1.84L5.25 8.051a.999.999 0 01.356-.257l4-1.714a1 1 0 11.788 1.838L7.667 9.088l1.94.831a1 1 0 00.787 0l7-3a1 1 0 000-1.838l-7-3zM3.31 9.397L5 10.12v4.102a8.969 8.969 0 00-1.05-.174 1 1 0 01-.89-.89 11.115 11.115 0 01.25-3.762zM9.3 16.573A9.026 9.026 0 007 14.935v-3.957l1.818.78a3 3 0 002.364 0l5.508-2.361a11.026 11.026 0 01.25 3.762 1 1 0 01-.89.89 8.968 8.968 0 00-5.35 2.524 1 1 0 01-1.4 0zM6 18a1 1 0 001-1v-2.065a8.935 8.935 0 00-2-.712V17a1 1 0 001 1z"/>
                    </svg>`;
                    break;
                case 'umkm':
                    color = '#7C3AED'; // Purple
                    // Menggunakan ikon yang sama dengan card UMKM (shopping cart)
                    iconHtml = `<svg style="color: white; width: 12px; height: 12px;" fill="currentColor" viewBox="0 0 20 20">
                        <path d="M3 1a1 1 0 000 2h1.22l.305 1.222a.997.997 0 00.01.042l1.358 5.43-.893.892C3.74 11.846 4.632 14 6.414 14H15a1 1 0 000-2H6.414l1-1H14a1 1 0 00.894-.553l3-6A1 1 0 0017 3H6.28l-.31-1.243A1 1 0 005 1H3zM16 16.5a1.5 1.5 0 11-3 0 1.5 1.5 0 013 0zM6.5 18a1.5 1.5 0 100-3 1.5 1.5 0 000 3z"/>
                    </svg>`;
                    break;
                case 'wisata':
                    color = '#EA580C'; // Orange
                    // Menggunakan ikon yang sama dengan card wisata (image/photo)
                    iconHtml = `<svg style="color: white; width: 12px; height: 12px;" fill="currentColor" viewBox="0 0 20 20">
                        <path fill-rule="evenodd" d="M4 3a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V5a2 2 0 00-2-2H4zm12 12H4l4-8 3 6 2-4 3 6z"/>
                    </svg>`;
                    break;
                default:
                    color = '#6B7280'; // Gray
                    iconHtml = '<i class="fas fa-map-marker" style="color: white; font-size: 12px;"></i>';
            }
            
            return L.divIcon({
                className: 'custom-marker',
                html: `
                    <div style="
                        background-color: ${color};
                        width: 30px;
                        height: 30px;
                        border-radius: 50% 50% 50% 0;
                        transform: rotate(-45deg);
                        border: 3px solid white;
                        box-shadow: 0 2px 10px rgba(0,0,0,0.3);
                        display: flex;
                        align-items: center;
                        justify-content: center;
                    ">
                        <div style="transform: rotate(45deg);">
                            ${iconHtml}
                        </div>
                    </div>
                `,
                iconSize: [30, 30],
                iconAnchor: [15, 30],
                popupAnchor: [0, -30]
            });
        }

        // Create popup content with image
        function createPopupContent(data) {
            return `
                <div style="min-width: 220px; max-width: 240px;">
                    <img src="${data.foto}" alt="${data.title}" 
                         style="width: 100%; height: 100px; object-fit: cover; border-radius: 6px; margin-bottom: 8px;"
                         onerror="this.src='https://via.placeholder.com/240x100/E5E7EB/9CA3AF?text=No+Image'">
                    
                    <h3 class="font-semibold text-gray-800 mb-2" style="margin: 0 0 6px 0; font-size: 14px; line-height: 1.3;">
                        ${data.title}
                    </h3>
                    
                    <p class="text-sm text-gray-600 mb-3" style="margin: 0 0 8px 0; line-height: 1.3; font-size: 12px;">
                        ${data.description}
                    </p>
                    
                    <div class="flex items-center justify-between text-xs text-gray-500 mb-2" style="margin-bottom: 6px;">
                        <span class="px-2 py-1 rounded-full text-white" style="background-color: ${getTypeColor(data.type)}; font-size: 10px; padding: 2px 6px;">
                            ${getTypeLabel(data.type)}
                        </span>
                        <span style="font-size: 10px;">
                            ${data.lat.toFixed(4)}, ${data.lng.toFixed(4)}
                        </span>
                    </div>
                    
                    <button onclick="openInGoogleMaps(${data.lat}, ${data.lng})" 
                            style="width: 100%; padding: 6px; background: #10B981; color: white; border: none; border-radius: 4px; font-size: 11px; cursor: pointer; margin-top: 4px;">
                        <i class="fas fa-directions" style="margin-right: 4px; font-size: 10px;"></i>
                        Buka di Google Maps
                    </button>
                </div>
            `;
        }

        // Get type color for badges
        function getTypeColor(type) {
            switch(type) {
                case 'kantor': return '#DC2626';
                case 'fasilitas': return '#2563EB';
                case 'umkm': return '#7C3AED';
                case 'wisata': return '#EA580C';
                default: return '#6B7280';
            }
        }

        // Get type label
        function getTypeLabel(type) {
            switch(type) {
                case 'kantor': return 'Kantor Desa';
                case 'fasilitas': return 'Fasilitas Umum';
                case 'umkm': return 'UMKM';
                case 'wisata': return 'Wisata';
                default: return 'Lainnya';
            }
        }

        // Open location in Google Maps
        function openInGoogleMaps(lat, lng) {
            window.open(`https://www.google.com/maps?q=${lat},${lng}`, '_blank');
        }

        function showLocationInfo(data) {
            const infoPanel = document.getElementById('infoPanel');
            const locationInfo = document.getElementById('locationInfo');
            
            locationInfo.innerHTML = `
                <div class="space-y-2">
                    <h4 class="font-medium text-gray-800">${data.title}</h4>
                    <p class="text-sm text-gray-600">${data.description}</p>
                    <div class="text-xs text-gray-500">
                        <div>Koordinat: ${data.lat.toFixed(6)}, ${data.lng.toFixed(6)}</div>
                        <div class="mt-1">Kategori: ${data.type.charAt(0).toUpperCase() + data.type.slice(1)}</div>
                    </div>
                </div>
            `;
            
            infoPanel.classList.remove('hidden');
        }

        function filterMarkers(type) {
            // Update active button
            document.querySelectorAll('.filter-btn').forEach(btn => {
                btn.classList.remove('active', 'bg-blue-600', 'text-white');
                btn.classList.add('bg-white', 'text-gray-600');
            });
            
            event.target.classList.add('active', 'bg-blue-600', 'text-white');
            event.target.classList.remove('bg-white', 'text-gray-600');
            
            // Hide all markers
            markers.forEach(marker => {
                map.removeLayer(marker);
            });
            
            // Show filtered markers
            const markersToShow = type === 'all' ? markerGroups.all : markerGroups[type];
            markersToShow.forEach(marker => {
                map.addLayer(marker);
            });
        }

        function toggleSearch() {
            const searchBar = document.getElementById('searchBar');
            const searchInput = document.getElementById('searchInput');
            
            if (searchBar.classList.contains('hidden')) {
                searchBar.classList.remove('hidden');
                setTimeout(() => searchInput.focus(), 100);
            } else {
                searchBar.classList.add('hidden');
                searchInput.value = '';
            }
        }

        function toggleLegend() {
            const legendPanel = document.getElementById('legendPanel');
            legendPanel.classList.toggle('hidden');
        }

        function searchLocation() {
            const query = document.getElementById('searchInput').value.toLowerCase();
            if (!query) return;
            
            // Simple search simulation (in real app, this would use geocoding API)
            const sampleLocations = {
                'kantor desa': [-7.257472, 112.637416],
                'puskesmas': [-7.258472, 112.638416],
                'sekolah': [-7.256472, 112.636416],
                'air terjun': [-7.260472, 112.640416]
            };
            
            for (const [name, coords] of Object.entries(sampleLocations)) {
                if (name.includes(query)) {
                    map.setView(coords, 17);
                    break;
                }
            }
        }

        function centerMap() {
            map.setView([-7.257472, 112.637416], 15);
        }

        function getCurrentLocation() {
            if (navigator.geolocation) {
                navigator.geolocation.getCurrentPosition(function(position) {
                    const lat = position.coords.latitude;
                    const lng = position.coords.longitude;
                    
                    map.setView([lat, lng], 17);
                    
                    // Add temporary marker for current location
                    const currentLocationMarker = L.marker([lat, lng])
                        .addTo(map)
                        .bindPopup('Lokasi Anda Saat Ini')
                        .openPopup();
                    
                    // Remove marker after 5 seconds
                    setTimeout(() => {
                        map.removeLayer(currentLocationMarker);
                    }, 5000);
                    
                }, function(error) {
                    alert('Tidak dapat mengakses lokasi Anda. Pastikan GPS aktif dan berikan izin akses lokasi.');
                });
            } else {
                alert('Browser Anda tidak mendukung layanan lokasi.');
            }
        }

        function toggleFullscreen() {
            const mapContainer = document.getElementById('mapContainer');
            
            if (!document.fullscreenElement) {
                mapContainer.requestFullscreen().then(() => {
                    mapContainer.style.height = '100vh';
                    mapContainer.style.maxHeight = 'none';
                    setTimeout(() => map.invalidateSize(), 200);
                }).catch((err) => {
                    console.log('Fullscreen not supported or denied');
                });
            } else {
                document.exitFullscreen().then(() => {
                    mapContainer.style.height = '50vh';
                    mapContainer.style.maxHeight = '450px';
                    setTimeout(() => map.invalidateSize(), 200);
                });
            }
        }

        // Search on Enter key
        document.addEventListener('DOMContentLoaded', function() {
            document.getElementById('searchInput')?.addEventListener('keypress', function(e) {
                if (e.key === 'Enter') {
                    searchLocation();
                }
            });
        });

        // Update statistics cards
        async function updateStatistics() {
            try {
                const [umkmResponse, wisataResponse, fasilitasResponse] = await Promise.all([
                    fetch('/api/v1/umkm-locations'),
                    fetch('/api/v1/wisata-locations'), 
                    fetch('/api/v1/fasilitas-locations')
                ]);

                const umkmData = await umkmResponse.json();
                const wisataData = await wisataResponse.json();
                const fasilitasData = await fasilitasResponse.json();

                // Update card statistics
                const fasilitasCount = fasilitasData.success ? fasilitasData.data.length : 0;
                const umkmCount = umkmData.success ? umkmData.data.length : 0;
                const wisataCount = wisataData.success ? wisataData.data.length : 0;

                console.log('Statistics:', { fasilitasCount, umkmCount, wisataCount });

                // Update DOM elements - target the cards by their order
                const statCards = document.querySelectorAll('.mobile-card .text-lg.font-bold');
                
                if (statCards.length >= 3) {
                    statCards[0].textContent = fasilitasCount; // Fasilitas Umum (first card)
                    statCards[1].textContent = umkmCount;      // UMKM (second card)
                    statCards[2].textContent = wisataCount;    // Wisata (third card)
                }

            } catch (error) {
                console.error('Error updating statistics:', error);
            }
        }

        // Initialize map when page loads
        document.addEventListener('DOMContentLoaded', function() {
            // Small delay to ensure container is fully rendered
            setTimeout(() => {
            initMap();
                updateStatistics();
                
                // Ensure map renders properly after initial load
                setTimeout(() => {
                    if (map) {
                        map.invalidateSize();
                    }
                }, 300);
            }, 100);
            
            // Check FontAwesome loading
            checkFontAwesome();
            
            // Ensure map renders properly after window resize
            window.addEventListener('resize', function() {
                setTimeout(() => {
                    if (map) {
                        map.invalidateSize();
                    }
                }, 100);
            });
        });
        
        // Check FontAwesome loading
        function checkFontAwesome() {
            setTimeout(() => {
                const testIcon = document.createElement('i');
                testIcon.className = 'fas fa-home';
                testIcon.style.visibility = 'hidden';
                testIcon.style.position = 'absolute';
                document.body.appendChild(testIcon);
                
                const computedStyle = window.getComputedStyle(testIcon, ':before');
                const content = computedStyle.getPropertyValue('content');
                
                if (content && content !== 'none' && content !== '""') {
                    console.log('✅ FontAwesome loaded successfully on peta page');
                } else {
                    console.error('❌ FontAwesome failed to load on peta page, adding fallback');
                    loadFontAwesomeFallback();
                }
                
                document.body.removeChild(testIcon);
            }, 1000);
        }
        
        // Load FontAwesome fallback
        function loadFontAwesomeFallback() {
            const link = document.createElement('link');
            link.rel = 'stylesheet';
            link.href = 'https://maxcdn.bootstrapcdn.com/font-awesome/4.7.0/css/font-awesome.min.css';
            link.onerror = function() {
                console.error('❌ All FontAwesome sources failed, showing text fallback');
                showTextFallback();
            };
            document.head.appendChild(link);
        }
        
        // Show text fallback for icons
        function showTextFallback() {
            const icons = document.querySelectorAll('.fas, .fab, .fa');
            icons.forEach(icon => {
                const className = icon.className;
                let text = '●';
                
                if (className.includes('fa-home')) text = '🏠';
                else if (className.includes('fa-newspaper')) text = '📰';
                else if (className.includes('fa-map-marked-alt')) text = '🗺️';
                else if (className.includes('fa-store')) text = '🏪';
                else if (className.includes('fa-map')) text = '🗺️';
                
                icon.innerHTML = text;
                icon.style.fontFamily = 'inherit';
                icon.style.fontSize = '16px';
            });
        }
        // Navigation function
        function navigateToService(serviceType) {
            console.log('Navigating to:', serviceType);
            
            switch(serviceType) {
                case '/':
                case 'home':
                case 'beranda':
                    window.location.href = '/';
                    break;
                case 'berita':
                    window.location.href = '/berita';
                    break;
                case 'wisata':
                    window.location.href = '/wisata';
                    break;
                case 'umkm':
                    window.location.href = '/produk-umkm';
                    break;
                case 'layanan-mandiri':
                    window.location.href = '/layanan-mandiri';
                    break;
                case 'peta':
                    window.location.href = '/peta';
                    break;
                default:
                    console.log('Unknown service type:', serviceType);
                    window.location.href = '/';
            }
        }
    </script>
</body>
</html>