<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\UmkmController;
use App\Http\Controllers\Api\BeritaController;
use App\Http\Controllers\Api\TourismController;
use App\Http\Controllers\Api\VillageController;
use App\Http\Controllers\Api\StaffController;
use App\Http\Controllers\Api\DemographicsController;
use App\Http\Controllers\Api\PPOBController;
use App\Http\Controllers\Api\MapController;
use App\Http\Controllers\Admin\ProdukUmkmController;
use App\Http\Controllers\Admin\ObjekWisataController;
use App\Http\Controllers\Admin\TiketWisataController;
use App\Http\Controllers\Admin\TransaksiController;
use App\Http\Controllers\PendudukController;
use App\Http\Controllers\BeritaController as WebBeritaController;
use App\Http\Controllers\PengaduanController;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Api\PortalWargaController;
use App\Http\Controllers\Api\WilayahController;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
|
| Here is where you can register API routes for your application. These
| routes are loaded by the RouteServiceProvider and all of them will
| be assigned to the "api" middleware group. Make something great!
|
*/

// =================== FLUTTER APP API ROUTES ===================
// Clean API routes for Flutter application with rate limiting
Route::group(['prefix' => '', 'middleware' => ['throttle:60,1']], function () {
    // Village Information
    Route::get('village-setting', [VillageController::class, 'getSetting']);
    Route::get('village-stats', [VillageController::class, 'getStats']);
    
    // Staff/Perangkat Desa
    Route::get('staff', [StaffController::class, 'index']);
    
    // News/Berita
    Route::get('berita', [BeritaController::class, 'index']);
    Route::get('berita/{id}', [BeritaController::class, 'show']);
    
    // UMKM
    Route::get('umkm', [UmkmController::class, 'index']);
    Route::get('umkm/{id}', [UmkmController::class, 'show']);
    Route::get('produk-umkm', [UmkmController::class, 'produkIndex']);
    Route::get('produk-umkm/{id}', [UmkmController::class, 'produkShow']);
    
    // Tourism/Wisata
    Route::get('wisata', [TourismController::class, 'index']);
    Route::get('wisata/{id}', [TourismController::class, 'show']);

    // Demographics
    Route::prefix('demografi')->group(function () {
        Route::get('overview', [DemographicsController::class, 'overview']);
        Route::get('age', [DemographicsController::class, 'byAge']);
        Route::get('education', [DemographicsController::class, 'byEducation']);
        Route::get('occupation', [DemographicsController::class, 'byOccupation']);
        Route::get('area', [DemographicsController::class, 'byArea']);
    });

    // PPOB
    Route::prefix('ppob')->group(function () {
        Route::get('services', [PPOBController::class, 'services']);
        Route::get('pulsa-products', [PPOBController::class, 'pulsaProducts']);
        Route::get('payment-methods', [PPOBController::class, 'paymentMethods']);
        Route::get('transactions', [PPOBController::class, 'transactions']);
        
        // Bill inquiry
        Route::post('electricity/inquiry', [PPOBController::class, 'inquireElectricity']);
        Route::post('water/inquiry', [PPOBController::class, 'inquireWater']);
        Route::post('tax/inquiry', [PPOBController::class, 'inquireTax']);
        
        // Payment processing
        Route::post('pulsa/purchase', [PPOBController::class, 'purchasePulsa']);
        Route::post('bill/pay', [PPOBController::class, 'payBill']);
    });

    // Map
    Route::prefix('map')->group(function () {
        Route::get('settings', [MapController::class, 'settings']);
        Route::get('locations', [MapController::class, 'locations']);
        Route::get('categories', [MapController::class, 'categories']);
        Route::get('areas', [MapController::class, 'areas']);
        Route::get('search', [MapController::class, 'search']);
    });

    // Portal Services (public)
    Route::prefix('portal')->group(function () {
        Route::get('services', [PortalWargaController::class, 'services']);
        Route::get('bantuan', [PortalWargaController::class, 'bantuanList']);
    });

    // Wilayah Administratif for registration form
    Route::prefix('wilayah')->group(function () {
        Route::get('dusun', [WilayahController::class, 'getDusun']);
        Route::get('rw/{dusunId}', [WilayahController::class, 'getRwByDusun']);
        Route::get('rt/{rwId}', [WilayahController::class, 'getRtByRw']);
        Route::get('registration', [WilayahController::class, 'getWilayahForRegistration']);
    });
});

// Protected routes for Flutter app
Route::middleware('auth:sanctum')->group(function () {
    // Tourism Booking
    Route::post('tiket-wisata', [TourismController::class, 'bookTicket']);
    Route::get('tiket-wisata', [TourismController::class, 'getTickets']);

    // Portal Protected Routes
    Route::prefix('portal')->group(function () {
        Route::get('profile', [PortalWargaController::class, 'profile']);
        Route::put('profile', [PortalWargaController::class, 'updateProfile']);
        Route::get('surat', [PortalWargaController::class, 'suratRequests']);
        Route::post('surat', [PortalWargaController::class, 'submitSuratRequest']);
        Route::get('pengaduan', [PortalWargaController::class, 'pengaduanList']);
        Route::post('pengaduan', [PortalWargaController::class, 'submitPengaduan']);
    });
});

// =================== LEGACY API ROUTES ===================
// Public routes
Route::prefix('v1')->group(function () {
    
    // Portal Warga API (no CSRF required)
    Route::post('portal/login', [PortalWargaController::class, 'login']);
    Route::post('portal/set-session', [\App\Http\Controllers\Api\PortalWargaController::class, 'setSession'])->name('portal.api.set-session');
    Route::post('portal/logout', [\App\Http\Controllers\Api\PortalWargaController::class, 'logoutWeb'])->name('portal.api.logout');
    Route::post('portal/test', function() {
        return response()->json([
            'status' => 'success',
            'message' => 'API connection test successful',
            'timestamp' => now()
        ]);
    });
    Route::post('portal/user-data', [PortalWargaController::class, 'getUserData']);
    Route::post('portal/family-data', [PortalWargaController::class, 'getFamilyData']);
    Route::post('portal/statistics', [PortalWargaController::class, 'getStatistics']);
    
    // Authentication
    Route::post('/auth/login', [AuthController::class, 'login']);
    Route::post('/auth/register', [AuthController::class, 'register']);
    Route::post('/auth/forgot-password', [AuthController::class, 'forgotPassword']);
    
    // OTP Authentication
    Route::post('/otp/send', [\App\Http\Controllers\Api\OtpController::class, 'sendOtp']);
    Route::post('/otp/verify', [\App\Http\Controllers\Api\OtpController::class, 'verifyOtp']);
    Route::post('/otp/resend', [\App\Http\Controllers\Api\OtpController::class, 'resendOtp']);
    Route::get('/otp/status', [\App\Http\Controllers\Api\OtpController::class, 'checkOtpStatus']);
    
    // NIK Verification
    Route::post('/verify-nik', [\App\Http\Controllers\Api\OtpController::class, 'verifyNik']);
    Route::get('/check-phone', [\App\Http\Controllers\Api\OtpController::class, 'checkPhoneRegistration']);
    
    // App Configuration
    Route::get('/config', [\App\Http\Controllers\Api\OtpController::class, 'getAppConfig']);
    
    // Public UMKM & Products
    Route::get('/umkm', [UmkmController::class, 'index']);
    Route::get('/umkm/{umkm}', [UmkmController::class, 'show']);
    Route::get('/umkm/{umkm}/produk', [UmkmController::class, 'produk']);
    
    Route::get('/produk', [ProdukUmkmController::class, 'index']);
    Route::get('/produk/{produk}', [ProdukUmkmController::class, 'show']);
    Route::get('/produk/kategori/{kategori}', [ProdukUmkmController::class, 'byKategori']);
    Route::get('/produk/featured', [ProdukUmkmController::class, 'featured']);
    Route::get('/produk/bestseller', [ProdukUmkmController::class, 'bestseller']);
    
    // Public Tourism
    Route::get('/wisata', [ObjekWisataController::class, 'index']);
    Route::get('/wisata/{objek}', [ObjekWisataController::class, 'show']);
    Route::get('/wisata/kategori/{kategori}', [ObjekWisataController::class, 'byKategori']);
    Route::get('/wisata/featured', [ObjekWisataController::class, 'featured']);
    Route::get('/wisata/nearby', [ObjekWisataController::class, 'nearby']);
    
    // Public News
    Route::get('/berita', [BeritaController::class, 'apiIndex']);
    Route::get('/berita/{berita}', [BeritaController::class, 'show']);
    Route::post('/berita/{id}/view', [BeritaController::class, 'incrementView']);
    Route::post('/berita/{id}/toggle-like', [BeritaController::class, 'toggleLike']);
    Route::post('/berita/{id}/increment-share', [BeritaController::class, 'incrementShare']);
    Route::get('/berita/{id}/images', [BeritaController::class, 'getImages']);
    
    // Comment route with web middleware for CSRF protection
    Route::middleware('web')->post('/berita/{id}/comment', [BeritaController::class, 'addComment']);
    
    // Public Population Data (limited)
    Route::get('/penduduk/statistik', [PendudukController::class, 'statistik']);
    
    // Public Perangkat Desa
    Route::get('/perangkat-desa', function () {
        try {
            // Add CORS headers
            header('Access-Control-Allow-Origin: *');
            header('Access-Control-Allow-Methods: GET, POST, OPTIONS');
            header('Access-Control-Allow-Headers: Content-Type, Authorization');
            
            $perangkat = \App\Models\PerangkatDesa::where('status', 'Aktif')
                                     ->orderByRaw("
                                         CASE jabatan
                                             WHEN 'Kepala Desa' THEN 1
                                             WHEN 'Sekretaris Desa' THEN 2
                                             WHEN 'Kaur Tata Usaha dan Umum' THEN 3
                                             WHEN 'Kaur Keuangan' THEN 4
                                             WHEN 'Kaur Perencanaan' THEN 5
                                             WHEN 'Kasi Pemerintahan' THEN 6
                                             WHEN 'Kasi Kesejahteraan' THEN 7
                                             WHEN 'Kasi Pelayanan' THEN 8
                                             WHEN 'Kepala Dusun' THEN 9
                                             ELSE 99
                                         END, nama
                                     ")
                                     ->limit(8)
                                     ->get();
            
            return response()->json([
                'success' => true,
                'data' => $perangkat,
                'message' => 'Data perangkat desa berhasil diambil',
                'total' => $perangkat->count()
            ])->header('Access-Control-Allow-Origin', '*')
              ->header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
              ->header('Access-Control-Allow-Headers', 'Content-Type, Authorization');
              
        } catch (\Exception $e) {
            Log::error('Perangkat Desa API Error: ' . $e->getMessage());
            
            return response()->json([
                'success' => false,
                'message' => 'Terjadi kesalahan saat mengambil data perangkat desa',
                'error' => $e->getMessage()
            ], 500)->header('Access-Control-Allow-Origin', '*')
              ->header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
              ->header('Access-Control-Allow-Headers', 'Content-Type, Authorization');
        }
    });
    
    // Public Setting Desa
    Route::get('/setting-desa', function () {
        $setting = \App\Models\Setting::first();
        
        return response()->json([
            'success' => true,
            'data' => [
                'nama_desa' => $setting->nama_desa ?? 'Desa Digital',
                'logo' => $setting->logo ? '/storage/' . $setting->logo : null,
                'kecamatan' => $setting->kecamatan ?? null,
                'kabupaten' => $setting->kabupaten ?? null,
                'provinsi' => $setting->provinsi ?? null
            ]
        ]);
    });
    
    // Map Location APIs
    Route::get('/umkm-locations', function () {
        // Try to get from database first
        $umkm = \App\Models\Umkm::whereNotNull('latitude')
                                ->whereNotNull('longitude')
                                ->where('status', 'aktif')
                                ->select('id', 'nama_umkm', 'kategori', 'alamat', 'latitude', 'longitude', 'logo as foto')
                                ->get();
        
        // If no data, return sample data
        if ($umkm->isEmpty()) {
            $umkm = collect([
                [
                    'id' => 1,
                    'nama_umkm' => 'Warung Makan Bu Sari',
                    'kategori' => 'Kuliner',
                    'alamat' => 'Jl. Desa Sejahtera No. 15',
                    'latitude' => -6.7175,
                    'longitude' => 107.1430,
                    'foto' => null
                ],
                [
                    'id' => 2,
                    'nama_umkm' => 'Kerajinan Bambu Kreatif',
                    'kategori' => 'Kerajinan',
                    'alamat' => 'Jl. Bamboo Raya No. 12',
                    'latitude' => -6.7180,
                    'longitude' => 107.1440,
                    'foto' => null
                ],
                [
                    'id' => 3,
                    'nama_umkm' => 'Tani Madu Makmur',
                    'kategori' => 'Pertanian',
                    'alamat' => 'Jl. Perkebunan No. 8',
                    'latitude' => -6.7170,
                    'longitude' => 107.1420,
                    'foto' => null
                ]
            ]);
        }
        
        return response()->json([
            'success' => true,
            'data' => $umkm
        ]);
    });
    
    Route::get('/wisata-locations', function () {
        // Try to get from database first
        $wisata = \App\Models\ObjekWisata::whereNotNull('latitude')
                                        ->whereNotNull('longitude')
                                        ->where('status', 'aktif')
                                        ->select('id', 'nama', 'deskripsi', 'alamat', 'latitude', 'longitude', 'foto')
                                        ->get();
        
        // If no data, return sample data
        if ($wisata->isEmpty()) {
            $wisata = collect([
                [
                    'id' => 1,
                    'nama' => 'Air Terjun Sirnajaya',
                    'deskripsi' => 'Air terjun alami dengan pemandangan yang indah',
                    'alamat' => 'Desa Sirnajaya',
                    'latitude' => -6.7190,
                    'longitude' => 107.1450,
                    'foto' => null
                ],
                [
                    'id' => 2,
                    'nama' => 'Kebun Teh Tradisional',
                    'deskripsi' => 'Agrowisata kebun teh dengan pemandangan pegunungan',
                    'alamat' => 'Kecamatan Sukamakmur',
                    'latitude' => -6.7160,
                    'longitude' => 107.1410,
                    'foto' => null
                ]
            ]);
        }
        
        return response()->json([
            'success' => true,
            'data' => $wisata
        ]);
    });
    
    Route::get('/fasilitas-locations', function () {
        // Static fasilitas data with realistic Sirnajaya locations
        $fasilitas = [
            [
                'id' => 1,
                'nama' => 'Kantor Desa Sirnajaya',
                'kategori' => 'Kantor Desa',
                'alamat' => 'Kampung Sirnajaya RT 01/RW 01, Desa Sirnajaya, Kec. Sukamakmur',
                'latitude' => -6.7175,
                'longitude' => 107.1426,
                'foto' => null,
                'jam_operasional' => 'Senin-Jumat: 08:00-16:00',
                'kontak' => '0263-123456'
            ],
            [
                'id' => 2,
                'nama' => 'Puskesmas Pembantu Sirnajaya',
                'kategori' => 'Kesehatan',
                'alamat' => 'Kampung Bojong RT 02/RW 01, Desa Sirnajaya, Kec. Sukamakmur',
                'latitude' => -6.7185,
                'longitude' => 107.1436,
                'foto' => null,
                'jam_operasional' => 'Senin-Sabtu: 07:00-14:00',
                'kontak' => '0263-123457'
            ],
            [
                'id' => 3,
                'nama' => 'SDN Sirnajaya 01',
                'kategori' => 'Pendidikan',
                'alamat' => 'Kampung Sirnajaya RT 02/RW 01, Desa Sirnajaya, Kec. Sukamakmur',
                'latitude' => -6.7165,
                'longitude' => 107.1416,
                'foto' => null,
                'jam_operasional' => 'Senin-Sabtu: 07:00-12:00',
                'kontak' => '0263-123458'
            ],
            [
                'id' => 4,
                'nama' => 'Masjid Al-Ikhlas Sirnajaya',
                'kategori' => 'Tempat Ibadah',
                'alamat' => 'Kampung Sirnajaya RT 01/RW 01, Desa Sirnajaya, Kec. Sukamakmur',
                'latitude' => -6.7170,
                'longitude' => 107.1430,
                'foto' => null,
                'jam_operasional' => '24 Jam',
                'kontak' => '081234567890'
            ],
            [
                'id' => 5,
                'nama' => 'Posyandu Melati',
                'kategori' => 'Kesehatan',
                'alamat' => 'Kampung Cigentis RT 01/RW 02, Desa Sirnajaya, Kec. Sukamakmur',
                'latitude' => -6.7190,
                'longitude' => 107.1420,
                'foto' => null,
                'jam_operasional' => 'Minggu ke-2 setiap bulan: 08:00-12:00',
                'kontak' => '081234567891'
            ],
            [
                'id' => 6,
                'nama' => 'Balai Desa Sirnajaya',
                'kategori' => 'Fasilitas Umum',
                'alamat' => 'Kampung Sirnajaya RT 01/RW 01, Desa Sirnajaya, Kec. Sukamakmur',
                'latitude' => -6.7178,
                'longitude' => 107.1428,
                'foto' => null,
                'jam_operasional' => 'Sesuai kebutuhan acara',
                'kontak' => '0263-123456'
            ],
            [
                'id' => 7,
                'nama' => 'Lapangan Olahraga Sirnajaya',
                'kategori' => 'Olahraga',
                'alamat' => 'Kampung Situ RT 02/RW 02, Desa Sirnajaya, Kec. Sukamakmur',
                'latitude' => -6.7160,
                'longitude' => 107.1440,
                'foto' => null,
                'jam_operasional' => '06:00-18:00',
                'kontak' => '081234567892'
            ],
            [
                'id' => 8,
                'nama' => 'TK Pertiwi Sirnajaya',
                'kategori' => 'Pendidikan',
                'alamat' => 'Kampung Bojong RT 02/RW 01, Desa Sirnajaya, Kec. Sukamakmur',
                'latitude' => -6.7168,
                'longitude' => 107.1418,
                'foto' => null,
                'jam_operasional' => 'Senin-Sabtu: 07:30-11:00',
                'kontak' => '081234567893'
            ]
        ];
        
        return response()->json([
            'success' => true,
            'data' => $fasilitas
        ]);
    });
    
    // Payment callback (public for Tripay)
    Route::post('/payment/callback', [TransaksiController::class, 'callback']);
    Route::get('/payment/return', [TransaksiController::class, 'return']);
    
});

// Protected routes
Route::prefix('v1')->middleware('auth:sanctum')->group(function () {
    
    // User profile
    Route::get('/user', function (Request $request) {
        return $request->user();
    });
    Route::post('/auth/logout', [AuthController::class, 'logout']);
    Route::put('/profile', [AuthController::class, 'updateProfile']);
    Route::post('/profile/avatar', [AuthController::class, 'updateAvatar']);
    
    // Portal Warga
    Route::get('/warga/dashboard', [\App\Http\Controllers\Api\PortalWargaController::class, 'dashboard']);
    Route::get('/warga/profile', [\App\Http\Controllers\Api\PortalWargaController::class, 'profile']);
    Route::put('/warga/profile', [\App\Http\Controllers\Api\PortalWargaController::class, 'updateProfile']);
    Route::get('/warga/notifications', [\App\Http\Controllers\Api\PortalWargaController::class, 'notifications']);
    
    // Layanan Surat Warga
    Route::get('/warga/surat', [\App\Http\Controllers\Api\PortalWargaController::class, 'layananSurat']);
    Route::post('/warga/surat', [\App\Http\Controllers\Api\PortalWargaController::class, 'ajukanSurat']);
    Route::get('/warga/surat/{nomorAntrian}/track', [\App\Http\Controllers\Api\PortalWargaController::class, 'trackSurat']);
    
    // Pengaduan Warga
    Route::get('/warga/pengaduan', [\App\Http\Controllers\Api\PortalWargaController::class, 'pengaduan']);
    Route::post('/warga/pengaduan', [\App\Http\Controllers\Api\PortalWargaController::class, 'submitPengaduan']);
    
    // User UMKM Management
    Route::get('/my/umkm', [UmkmController::class, 'myUmkm']);
    Route::post('/my/umkm', [UmkmController::class, 'store']);
    Route::put('/my/umkm/{umkm}', [UmkmController::class, 'update']);
    Route::delete('/my/umkm/{umkm}', [UmkmController::class, 'destroy']);
    
    // User Product Management
    Route::get('/my/produk', [ProdukUmkmController::class, 'myProduk']);
    Route::post('/my/produk', [ProdukUmkmController::class, 'store']);
    Route::put('/my/produk/{produk}', [ProdukUmkmController::class, 'update']);
    Route::delete('/my/produk/{produk}', [ProdukUmkmController::class, 'destroy']);
    Route::post('/my/produk/{produk}/upload-foto', [ProdukUmkmController::class, 'uploadFoto']);
    
    // Ticket Booking
    Route::post('/tiket/book', [TiketWisataController::class, 'book']);
    Route::get('/my/tiket', [TiketWisataController::class, 'myTickets']);
    Route::get('/my/tiket/{tiket}', [TiketWisataController::class, 'show']);
    Route::post('/my/tiket/{tiket}/cancel', [TiketWisataController::class, 'cancel']);
    Route::post('/my/tiket/{tiket}/review', [TiketWisataController::class, 'addReview']);
    
    // Transactions
    Route::get('/my/transaksi', [TransaksiController::class, 'myTransactions']);
    Route::get('/my/transaksi/{transaksi}', [TransaksiController::class, 'show']);
    Route::post('/transaksi/create', [TransaksiController::class, 'create']);
    Route::get('/transaksi/{transaksi}/status', [TransaksiController::class, 'checkStatus']);
    
    // Complaints
    Route::get('/my/pengaduan', [PengaduanController::class, 'myComplaints']);
    Route::post('/pengaduan', [PengaduanController::class, 'store']);
    Route::put('/pengaduan/{pengaduan}', [PengaduanController::class, 'update']);
    
    // Wishlist & Favorites
    Route::get('/my/wishlist', [ProdukUmkmController::class, 'wishlist']);
    Route::post('/produk/{produk}/wishlist', [ProdukUmkmController::class, 'toggleWishlist']);
    Route::get('/my/favorites', [ObjekWisataController::class, 'favorites']);
    Route::post('/wisata/{objek}/favorite', [ObjekWisataController::class, 'toggleFavorite']);
    
    // Reviews & Ratings
    Route::post('/produk/{produk}/review', [ProdukUmkmController::class, 'addReview']);
    Route::post('/wisata/{objek}/review', [ObjekWisataController::class, 'addReview']);
    
});

// Admin routes
Route::prefix('v1/admin')->middleware(['auth:sanctum', 'role:admin'])->group(function () {
    
    // UMKM Management
    Route::apiResource('umkm', UmkmController::class);
    Route::put('umkm/{umkm}/verify', [UmkmController::class, 'verify']);
    Route::put('umkm/{umkm}/toggle-status', [UmkmController::class, 'toggleStatus']);
    Route::get('umkm/pending/verification', [UmkmController::class, 'pendingVerification']);
    
    // Product Management
    Route::apiResource('produk', ProdukUmkmController::class);
    Route::put('produk/{produk}/toggle-status', [ProdukUmkmController::class, 'toggleStatus']);
    Route::put('produk/{produk}/toggle-featured', [ProdukUmkmController::class, 'toggleFeatured']);
    Route::post('produk/bulk-action', [ProdukUmkmController::class, 'bulkAction']);
    
    // Tourism Management
    Route::apiResource('wisata', ObjekWisataController::class);
    Route::put('wisata/{objek}/toggle-status', [ObjekWisataController::class, 'toggleStatus']);
    Route::put('wisata/{objek}/toggle-featured', [ObjekWisataController::class, 'toggleFeatured']);
    Route::get('wisata/statistik/dashboard', [ObjekWisataController::class, 'dashboardStats']);
    
    // Ticket Management
    Route::apiResource('tiket', TiketWisataController::class);
    Route::post('tiket/{tiket}/check-in', [TiketWisataController::class, 'checkIn']);
    Route::post('tiket/{tiket}/check-out', [TiketWisataController::class, 'checkOut']);
    Route::get('tiket/scan/{qrCode}', [TiketWisataController::class, 'scanQrCode']);
    Route::get('tiket/today/visitors', [TiketWisataController::class, 'todayVisitors']);
    
    // Transaction Management
    Route::apiResource('transaksi', TransaksiController::class);
    Route::post('transaksi/{transaksi}/refund', [TransaksiController::class, 'refund']);
    Route::put('transaksi/{transaksi}/verify', [TransaksiController::class, 'verify']);
    Route::get('transaksi/laporan/harian', [TransaksiController::class, 'laporanHarian']);
    Route::get('transaksi/laporan/bulanan', [TransaksiController::class, 'laporanBulanan']);
    
    // Statistics & Reports
    Route::get('dashboard/stats', [UmkmController::class, 'dashboardStats']);
    Route::get('reports/umkm', [UmkmController::class, 'reportUmkm']);
    Route::get('reports/wisata', [ObjekWisataController::class, 'reportWisata']);
    Route::get('reports/transaksi', [TransaksiController::class, 'reportTransaksi']);
    
});

// Utility routes
Route::prefix('v1/utils')->group(function () {
    
    // Location services
    Route::get('/provinces', function() {
        return response()->json(['data' => ['Jawa Timur', 'Jawa Tengah', 'Jawa Barat', 'DKI Jakarta']]);
    });
    
    Route::get('/cities/{province}', function($province) {
        $cities = [
            'Jawa Timur' => ['Surabaya', 'Malang', 'Kediri', 'Blitar', 'Tulungagung'],
            'Jawa Tengah' => ['Semarang', 'Solo', 'Yogyakarta', 'Magelang'],
            'Jawa Barat' => ['Bandung', 'Bogor', 'Depok', 'Bekasi'],
            'DKI Jakarta' => ['Jakarta Pusat', 'Jakarta Utara', 'Jakarta Selatan', 'Jakarta Timur', 'Jakarta Barat']
        ];
        
        return response()->json(['data' => $cities[$province] ?? []]);
    });
    
    // Categories
    Route::get('/categories/umkm', function() {
        return response()->json([
            'data' => [
                'makanan_minuman' => 'Makanan & Minuman',
                'kerajinan' => 'Kerajinan',
                'fashion' => 'Fashion',
                'pertanian' => 'Pertanian',
                'jasa' => 'Jasa',
                'lainnya' => 'Lainnya'
            ]
        ]);
    });
    
    Route::get('/categories/produk', function() {
        return response()->json([
            'data' => [
                'makanan_segar' => 'Makanan Segar',
                'makanan_olahan' => 'Makanan Olahan',
                'minuman' => 'Minuman',
                'kerajinan_kayu' => 'Kerajinan Kayu',
                'kerajinan_bambu' => 'Kerajinan Bambu',
                'fashion_pakaian' => 'Fashion Pakaian',
                'elektronik' => 'Elektronik',
                'lainnya' => 'Lainnya'
            ]
        ]);
    });
    
    Route::get('/categories/wisata', function() {
        return response()->json([
            'data' => [
                'alam' => 'Wisata Alam',
                'budaya' => 'Wisata Budaya',
                'sejarah' => 'Wisata Sejarah',
                'religi' => 'Wisata Religi',
                'kuliner' => 'Wisata Kuliner',
                'adventure' => 'Wisata Adventure',
                'lainnya' => 'Lainnya'
            ]
        ]);
    });
    
    // Payment methods
    Route::get('/payment-methods', function() {
        return response()->json([
            'data' => [
                'virtual_account' => [
                    'MYBVA' => 'Maybank VA',
                    'PERMATAVA' => 'Permata VA',
                    'BNIVA' => 'BNI VA',
                    'BRIVA' => 'BRI VA',
                    'MANDIRIVA' => 'Mandiri VA',
                    'BCAVA' => 'BCA VA',
                ],
                'e_wallet' => [
                    'OVO' => 'OVO',
                    'DANA' => 'DANA',
                    'SHOPEEPAY' => 'ShopeePay',
                    'LINKAJA' => 'LinkAja',
                ],
                'retail' => [
                    'ALFAMART' => 'Alfamart',
                    'INDOMARET' => 'Indomaret',
                ],
                'qris' => [
                    'QRIS' => 'QRIS',
                ]
            ]
        ]);
    });
    
});

// Statistik dashboard endpoint
Route::get('/v1/statistik-dashboard', function() {
    try {
        $totalPenduduk = \App\Models\Penduduk::count();
        $totalKeluarga = \App\Models\Keluarga::count();
        
        return response()->json([
            'success' => true,
            'data' => [
                'total_penduduk' => $totalPenduduk,
                'total_keluarga' => $totalKeluarga
            ]
        ]);
        
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'error' => $e->getMessage()
        ], 500);
    }
});

// Endpoint untuk melihat struktur tabel penduduk
Route::get('/v1/penduduk/struktur', function() {
    try {
        // Get table structure
        $columns = DB::select('DESCRIBE penduduk');
        
        // Get sample data
        $sampleData = \App\Models\Penduduk::take(3)->get();
        
        // Get statistics
        $stats = [
            'total_records' => \App\Models\Penduduk::count(),
            'columns_count' => count($columns),
            'table_size' => DB::select("SELECT 
                ROUND(((data_length + index_length) / 1024 / 1024), 2) AS size_mb 
                FROM information_schema.tables 
                WHERE table_schema = DATABASE() 
                AND table_name = 'penduduk'")[0]->size_mb ?? 0
        ];
        
        return response()->json([
            'success' => true,
            'data' => [
                'table_name' => 'penduduk',
                'columns' => $columns,
                'statistics' => $stats,
                'sample_data' => $sampleData->take(2) // Only show 2 samples for brevity
            ]
        ]);
        
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'error' => $e->getMessage()
        ], 500);
    }
});

// Endpoint untuk manajemen penduduk dengan kolom baru
Route::prefix('v1/penduduk')->group(function() {
    
    // Mapping nama orang tua ke NIK
    Route::post('/map-orangtua', function() {
        try {
            $results = \App\Models\Penduduk::batchMapOrangTua();
            
            return response()->json([
                'success' => true,
                'message' => 'Mapping orang tua berhasil dilakukan',
                'data' => $results
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    });
    
    // Preview mapping sebelum eksekusi
    Route::get('/preview-mapping', function() {
        try {
            $pendudukNeedMapping = \App\Models\Penduduk::where(function($query) {
                $query->where(function($q) {
                    $q->whereNotNull('nama_ayah')
                      ->whereNull('ayah_nik');
                })->orWhere(function($q) {
                    $q->whereNotNull('nama_ibu')
                      ->whereNull('ibu_nik');
                });
            })->take(20)->get(['nik', 'nama', 'nama_ayah', 'nama_ibu', 'ayah_nik', 'ibu_nik']);
            
            $preview = $pendudukNeedMapping->map(function($penduduk) {
                return [
                    'nik' => $penduduk->nik,
                    'nama' => $penduduk->nama,
                    'nama_ayah' => $penduduk->nama_ayah,
                    'nama_ibu' => $penduduk->nama_ibu,
                    'nik_ayah_found' => $penduduk->nama_ayah ? \App\Models\Penduduk::findNikByNama($penduduk->nama_ayah, 'L') : null,
                    'nik_ibu_found' => $penduduk->nama_ibu ? \App\Models\Penduduk::findNikByNama($penduduk->nama_ibu, 'P') : null
                ];
            });
            
            return response()->json([
                'success' => true,
                'data' => [
                    'total_need_mapping' => \App\Models\Penduduk::where(function($query) {
                        $query->where(function($q) {
                            $q->whereNotNull('nama_ayah')
                              ->whereNull('ayah_nik');
                        })->orWhere(function($q) {
                            $q->whereNotNull('nama_ibu')
                              ->whereNull('ibu_nik');
                        });
                    })->count(),
                    'preview' => $preview
                ]
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    });
    
    // Statistik akta lahir
    Route::get('/stats-akta', function() {
        try {
            $stats = [
                'total_penduduk' => \App\Models\Penduduk::count(),
                'has_akta' => \App\Models\Penduduk::hasAktaLahir()->count(),
                'without_akta' => \App\Models\Penduduk::withoutAktaLahir()->count(),
                'akta_percentage' => 0
            ];
            
            if ($stats['total_penduduk'] > 0) {
                $stats['akta_percentage'] = round(($stats['has_akta'] / $stats['total_penduduk']) * 100, 2);
            }
            
            return response()->json([
                'success' => true,
                'data' => $stats
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    });
    
    // Pencarian penduduk dengan filter baru
    Route::get('/search', function(\Illuminate\Http\Request $request) {
        try {
            $query = \App\Models\Penduduk::query();
            
            // Filter berdasarkan nama
            if ($request->has('nama')) {
                $query->where('nama', 'LIKE', '%' . $request->nama . '%');
            }
            
            // Filter berdasarkan nama orang tua
            if ($request->has('nama_orangtua')) {
                $query->byNamaOrangTua($request->nama_orangtua);
            }
            
            // Filter berdasarkan akta lahir
            if ($request->has('has_akta')) {
                if ($request->has_akta === 'true') {
                    $query->hasAktaLahir();
                } else {
                    $query->withoutAktaLahir();
                }
            }
            
            // Filter berdasarkan dusun
            if ($request->has('dusun')) {
                $query->byDusun($request->dusun);
            }
            
            $penduduk = $query->with(['ayah:nik,nama', 'ibu:nik,nama'])
                            ->paginate($request->get('per_page', 15));
            
            return response()->json([
                'success' => true,
                'data' => $penduduk
            ]);
            
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    });
});

// Performance Management Routes
Route::prefix('v1/admin/performance')->middleware(['auth:sanctum'])->group(function () {
    Route::get('/metrics', [App\Http\Controllers\Admin\PerformanceController::class, 'getMetrics']);
    Route::post('/optimize-database', [App\Http\Controllers\Admin\PerformanceController::class, 'optimizeDatabase']);
});

// Quick Performance Test Route
Route::get('/v1/performance/test', function() {
    try {
        $serviceClass = '\App\Services\PerformanceOptimizationService';
        if (class_exists($serviceClass)) {
            $metrics = $serviceClass::getDatabaseMetrics();
        } else {
            $metrics = ['error' => 'PerformanceOptimizationService not found'];
        }
        
        return response()->json([
            'success' => true,
            'data' => $metrics,
            'message' => 'Performance service working correctly'
        ]);
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'error' => $e->getMessage()
        ], 500);
    }
});

// Portal test route (no CSRF required)
// Public Statistics API untuk homepage
Route::get('/statistik-desa', function () {
    try {
        $stats = [
            'total_penduduk' => \App\Models\Penduduk::count(),
            'total_keluarga' => \App\Models\Penduduk::distinct('no_kk')->count('no_kk'),
            'total_surat' => \App\Models\PelayananSurat::count(),
            'total_pengaduan' => \App\Models\Pengaduan::count(),
            
            // Statistik Mutasi Penduduk
            'lahir_bulan_ini' => \App\Models\MutasiPenduduk::where('jenis_mutasi', 'Lahir')->whereMonth('tanggal_mutasi', now()->month)->whereYear('tanggal_mutasi', now()->year)->count(),
            'meninggal_bulan_ini' => \App\Models\MutasiPenduduk::where('jenis_mutasi', 'Meninggal')->whereMonth('tanggal_mutasi', now()->month)->whereYear('tanggal_mutasi', now()->year)->count(),
            'pindah_masuk_bulan_ini' => \App\Models\MutasiPenduduk::where('jenis_mutasi', 'Pindah Masuk')->whereMonth('tanggal_mutasi', now()->month)->whereYear('tanggal_mutasi', now()->year)->count(),
            'pindah_keluar_bulan_ini' => \App\Models\MutasiPenduduk::where('jenis_mutasi', 'Pindah Keluar')->whereMonth('tanggal_mutasi', now()->month)->whereYear('tanggal_mutasi', now()->year)->count(),
            
            // Total mutasi tahun ini
            'total_lahir_tahun_ini' => \App\Models\MutasiPenduduk::where('jenis_mutasi', 'Lahir')->whereYear('tanggal_mutasi', now()->year)->count(),
            'total_meninggal_tahun_ini' => \App\Models\MutasiPenduduk::where('jenis_mutasi', 'Meninggal')->whereYear('tanggal_mutasi', now()->year)->count(),
            'total_pindah_masuk_tahun_ini' => \App\Models\MutasiPenduduk::where('jenis_mutasi', 'Pindah Masuk')->whereYear('tanggal_mutasi', now()->year)->count(),
            'total_pindah_keluar_tahun_ini' => \App\Models\MutasiPenduduk::where('jenis_mutasi', 'Pindah Keluar')->whereYear('tanggal_mutasi', now()->year)->count(),
        ];
        
        return response()->json([
            'success' => true,
            'data' => $stats
        ])->header('Access-Control-Allow-Origin', '*')
          ->header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
          ->header('Access-Control-Allow-Headers', 'Content-Type, Authorization');
          
    } catch (\Exception $e) {
        return response()->json([
            'success' => false,
            'message' => 'Error loading statistics',
            'error' => $e->getMessage()
        ], 500)->header('Access-Control-Allow-Origin', '*')
          ->header('Access-Control-Allow-Methods', 'GET, POST, OPTIONS')
          ->header('Access-Control-Allow-Headers', 'Content-Type, Authorization');
    }
});

Route::post('v1/portal/test', function(\Illuminate\Http\Request $request) {
    return response()->json([
        'success' => true,
        'message' => 'API routing working (no CSRF required)',
        'timestamp' => now(),
        'method' => $request->method(),
        'url' => $request->url(),
        'ip' => $request->ip(),
        'user_agent' => $request->userAgent()
    ]);
}); 